/*
 * packet.cpp - Packet - this is what it all comes down to :)
 *
 * Written by Michael "Mr. Sump" Poppitz.
 *
 * Please note that this packet implementation does not have the
 * iso3309 checksum field, since this is taken care of by the linux kernel.
 * If you want to port this code to another platform you might need
 * to implement checksum routines.
 *
 */

 /***************************************************************************
  *                                                                         *
  *   This program is free software; you can redistribute it and/or modify  *
  *   it under the terms of the GNU General Public License as published by  *
  *   the Free Software Foundation; either version 2 of the License, or     *
  *   (at your option) any later version.                                   *
  *                                                                         *
  ***************************************************************************/

#include <stdio.h>
#include "packet.h"

/**** constructor / destructor *****************************************/

Packet::Packet(void *r, int len) {
	raw = (unsigned char *)r;
	data = raw + 1;
	length = len - getBaseLength();
	strcpy(dev, "");
}

Packet::Packet() {
	raw = (unsigned char *)calloc(DATA_LIMIT + 1, 1);
	data = raw + 1;
	length = 0;
	strcpy(dev, "");
	
	// set default values
	for (int i = 0; i < DATA_LIMIT; i++)
		data[i] = 0;
	data[6] |= 0x60;
	data[13] |= 0x60;
	
}

Packet::~Packet() {
	if (raw != NULL)
		free(raw);
}

/**** low level data operations ****************************************/

void Packet::setDevice(char *device) {
	strcpy(dev, device);
}

void Packet::getDevice(char *device) {
	strcpy(device, dev);
}

int Packet::setData(void *src, int len) {
	if (len <= 0 || len > DATA_LIMIT + 1)
		return (ERROR);

	memcpy((void *)raw, src, len);
	setDataLength(len);

	return (len);
}

int Packet::getData(void *dest) {
	int l = getDataLength();

	if (raw == NULL || l == 0)
		return (ERROR);

	memcpy(dest, (void *)raw, l);

	return (l);
}

void *Packet::getDataPointer() {
	return (raw);
}

int Packet::getDataLength() {
	return (getLength() + 1);
}

void Packet::setDataLength(int len) {
	length = len - getBaseLength() -1;
}

/**** retrieving information from packet *******************************/

Call *Packet::getSourceCall(void) {
	char callbuffer[7];
	int ssid = decodeCall(callbuffer, data + 7);
	return (new Call(callbuffer, ssid));
}

Call *Packet::getDestinationCall(void) {
	char callbuffer[7];
	int ssid = decodeCall(callbuffer, data);
	return (new Call(callbuffer, ssid));
}

Call *Packet::getRepeaterCall(int nr) {
	if (!isValidRepeaterNr(nr))
		return (NULL);

	char callbuffer[7];
	int ssid = decodeCall(callbuffer, data + 7 * (nr + 2));
	return (new Call(callbuffer, ssid));
}

bool Packet::hasBeenRepeated(int nr) {
	if (!isValidRepeaterNr(nr))
		return (false);

	return (data[7 * (nr + 2) + 6] & 0x80);
}
	
int Packet::getRepeaterCount() {
	int count = 0;

	while((!(data[7 * (count + 2) - 1] & 0x01)) && (count < REPEATER_LIMIT))
		count++;

	return (count);
}

int Packet::getFrameType() {
	switch (getControlByte() & 0x03) {
		case 0x00:
		case 0x02: return (FRAME_I);
		case 0x01: return (FRAME_S);
		case 0x03: return (FRAME_U);
		default: return (ERROR);
	}
}

int Packet::getSFrameType() {
	if (getFrameType() != FRAME_S)
		return (ERROR);

	int v = (int )((getControlByte() & 0x0c) >> 2);
	if (v > 2)
		return (ERROR);
	return (v);
}

int Packet::getUFrameType() {
	unsigned char ctl = getControlByte();
	unsigned char res = ((ctl & 0xe0) >> 3) | ((ctl & 0x0c) >> 2);
	switch (res) {
		case UFRAME_UI:
		case UFRAME_DM:
		case UFRAME_SABM:
		case UFRAME_DISC:
		case UFRAME_UA:
		case UFRAME_FRMR:
			return (res);
		default: return (ERROR);
	}
}


int Packet::getReceiveSequence() {
	if (getFrameType() == FRAME_U)
		return (ERROR);

	return ((int )((getControlByte() & 0xe0) >> 5));
}

int Packet::getSendSequence() {
	if (getFrameType() != FRAME_I)
		return (ERROR);

	return ((int )((getControlByte() & 0x0e) >> 1));
}

int Packet::getPID() {
	if (!hasInfo())
		return (ERROR);

	return ((int )data[7 * (2 + getRepeaterCount()) + 1]);
}

bool Packet::hasInfo() {
	int ft = getFrameType();
	return (ft == FRAME_I || (ft == FRAME_U && getUFrameType() == UFRAME_UI));
}

int Packet::getInfo(unsigned char *buffer) {
	int sze = getInfoLength();
	if (sze < 0 || sze > 0xff)
		return (ERROR);

	memcpy((void *)buffer, (void *)(data + 7 * (2 + getRepeaterCount()) + 2), sze);
	return (sze);
}

int Packet::getInfoLength() {
	return (length);
}

int Packet::getLength(void) {
	if (hasInfo()) {
		return (getBaseLength() + length);
	} else {
		return (getBaseLength());
	}
}

/**** writing information to packet ************************************/

// switch receiver & sender calls as well as repeater order and repeated flag
void Packet::reverseCalls(void) {
	char buf[7];
	void *buffer = (void *)buf;

	// switch source & destination
	memcpy(buffer, data, 7);
	memcpy(data, data + 7, 7);
	memcpy(data + 7, buffer, 7);

	// fix repeater flag
	if (data[6] & 0x01) {
		data[13] |= 0x01;
		data[6] &= 0xfe;
	}

	// mirror repeater order and inverse has-been-repeated flags
	int cnt = getRepeaterCount();
	if (cnt > 0) {
		int swcnt = cnt / 2;

		// switch repeater
		for (int i = 0; i < swcnt; i++) {
			void *a = data + 7 * (2 + i);
			void *b = data + 7 * (1 + cnt - i);
			memcpy(buffer, a, 7);
			memcpy(a, b, 7);
			memcpy(b, buffer, 7);
		}

		// negate has-been-repeated flags
		for (int i = 0; i < cnt; i++)
			data[7 * (2 + i) + 6] ^= 0x80;
				
		// repair repeater flag
		data[7 * 2 + 6] &= 0xfe;
		data[7 * (1 + cnt) + 6] |= 0x01;
	}
}

void Packet::setSourceCall(Call *call) {
	encodeCall(data + 7, call->call, call->ssid);
}

void Packet::setDestinationCall(Call *call) {
	encodeCall(data, call->call, call->ssid);
}

void Packet::setRepeaterCall(int nr, Call *call) {
	encodeCall(data + 7 * (2 + nr), call->call, call->ssid);
}

bool Packet::setRepeaterCount(int num) {
	if (num < 0 || num > REPEATER_LIMIT)
		return (false);

	// get current repeater count and if equal to given one, return success
	int c = getRepeaterCount();
	if (c == num)
		return (true);

	// move data using overlap proof memmove
	memmove(
		(void *)(data + 7 * (2 + num)),
		(void *)(data + 7 * (2 + c)),
		getLength() - 7 * (2 + c)
	);

	// set chain flags properly
	for (int i = 0; i < num; i++)
		data[7 * (i + 2) - 1] &= 0xfe;

	data[7 * (num + 2) - 1] |= 0x01;

	return (true);
}

void Packet::setCommand() {
	data[6] |= 0x80;
	data[13] &= 0x7f;
}

void Packet::setResponse() {
	data[6] &= 0x7f;
	data[13] |= 0x80;
}

void Packet::disableCR() {
	data[6] &=0x7f;
	data[13] &=0x7f;
}    

bool Packet::setFrameType(int type, int subtype) {
	unsigned char *ctl = data + 7 * (2 + getRepeaterCount());

	switch (type) {
		case FRAME_I:
			ctl[0] &= 0xfe;
			ctl[1] = subtype;
			break;

		case FRAME_S:
			ctl[0] = (ctl[0] & 0xfd) | 0x01;
			switch (subtype) {
				case SFRAME_RR:   ctl[0] &= 0xf3;
				case SFRAME_RNR:  ctl[0] = (ctl[0] & 0xf7) | 0x04; break;
				case SFRAME_REJ:  ctl[0] = (ctl[0] & 0xfb) | 0x08; break;
				default: return (false);
			}
			break;

		case FRAME_U:
			ctl[0] |= 0x03;
			switch (subtype) {
				case UFRAME_UI:   ctl[0] = 0x13; break;
				case UFRAME_DM:   ctl[0] = 0x1f; break;
				case UFRAME_SABM: ctl[0] = 0x3f; break;
				case UFRAME_DISC: ctl[0] = 0x53; break;
				case UFRAME_UA:   ctl[0] = 0x73; break;
				case UFRAME_FRMR: ctl[0] = 0x97; break;
				default: return (false);
			}
			break;

		default: return (false); 
	}
	return (true);
}

bool Packet::setReceiveSequence(int seq) {
	if (getFrameType() == FRAME_U)
		return (false);

	data[14 + 7 * getRepeaterCount()] &= 0x1f;
	data[14 + 7 * getRepeaterCount()] |= ((seq & 0x07) << 5);

	return (true);
}

bool Packet::setSendSequence(int seq) {
	if (getFrameType() == FRAME_U)
		return (false);

	data[14 + 7 * getRepeaterCount()] &= 0xf1;
	data[14 + 7 * getRepeaterCount()] |= ((seq & 0x07) << 1);

	return (true);
}

bool Packet::setInfo(unsigned char *buff, int len) {
	if (!hasInfo())
		return (false);

	memcpy((void *)(data + getBaseLength()), (void *)buff, len);
	length = len;

	return (true);
}

/**** helper ***********************************************************/

int Packet::decodeCall(char *decall, unsigned char *encall) {
	for (int i = 0; i < 6; i++) {
		decall[i] = (char)((encall[i] >> 1) & 0x7F);
		if (decall[i] == ' ')
			decall[i] = '\0';
	}
	decall[6] = '\0';

	return ((encall[6] & 0x1E ) >> 1);
}

void Packet::encodeCall(unsigned char *encall, char *decall, int ssid) {
	for (int i = 0; i < 6 && decall[i] != '\0'; i++) {
		encall[i] &= 0x01;
		encall[i] |= ((decall[i] & 0x7F) << 1);
	}
	for (int i = strlen(decall); i < 6; i++) {
		encall[i] &= 0x01;
		encall[i] |= (' ' << 1);
	}
	encall[6] &= 0xE1;
	encall[6] |= (ssid & 0x0F) << 1;
} 

unsigned char Packet::getControlByte() {
	return (data[7 * (2 + getRepeaterCount())]);
}


int Packet::getBaseLength() {
	int l = 7 * (2 + getRepeaterCount());

	if (hasInfo()) {
		l += 2;
	} else {
		l += 1;
	}
	return (l);
}

bool Packet::isValidRepeaterNr(int nr) {
	if (nr < 0)
		return (false);
	if (nr >= getRepeaterCount())
		return (false);

	return (true);
}
