/*
 * HTTPSniff v0.3 - HTTP Response Sniffer
 * main.c
 * (C)2004-2005 Michael Poppitz
 *
 * This file contains the command line parser and help page.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"
#include "sniff.h"
#include "nids.h"

#define MODE_AUTO 0
#define MODE_FILE 1
#define MODE_DEV 2

#define STACK_INTERNAL 0
#define STACK_NIDS 1

void mainHelp(char *name) {
	printf(
		"\n"
		"Usage: %s [<options>] <name> [<filter>]\n"
		"\n"
		"Where <options> are:\n"
		"	-i	force <name> to be treated as network interface\n"
		"	-f	force <name> to be treated as dump file\n"
		"\n"
		"	-x	use builtin relaxed tcp/ip stack instead of libnids (unsafe)\n"
		"	-u	use URL based file/directory structure for saved files (unsafe)\n"
		"\n"
		"	-v	verbose - additional v increase output further\n"
		"	-q	quiet - additional q decrease output further\n"
		"	-h	this help page\n"
		"\n"
		"The optional <filter> can be any pcap filter program. See tcpdump man page for\n"
		"details.\n"
		"\n"
		,
		name
	);
	exit(EXIT_FAILURE);
}

int main(int argc, char **argv) {
	int i, j, result, mode, stack, filterSize = 1;
	char *name = NULL, *filter = NULL;
	
	/* set defaults */
	debug = 1;
	cfg.saveStructured = 0;
	mode = MODE_AUTO;
	stack = STACK_NIDS;

	/* parse arguments */
	if (argc < 2)
		mainHelp(argv[0]);
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			for (j = 1; j < strlen(argv[i]); j++) {
				switch (argv[i][j]) {
					case 'f':
						if (mode == MODE_DEV) {
							printf("FATAL: -f and -i must not be used together.\n");
							mainHelp(argv[0]);
						}
						mode = MODE_FILE;
						break;
					
					case 'h':
						mainHelp(argv[0]);
						break;
	
					case 'i':
						if (mode == MODE_FILE) {
							printf("FATAL: -f and -i must not be used together.\n");
							mainHelp(argv[0]);
						}
						mode = MODE_DEV;
						break;

					case 'x':
						stack = STACK_INTERNAL;
						break;
					
					case 'u':
						cfg.saveStructured = 1;
						break;

					case 'q':
						if (debug > 0) debug--;
						break;

					case 'v':
						debug++;
						break;
	
					default:
						printf("FATAL: Invalid option: %s\n", argv[i]);
						mainHelp(argv[0]);
						break;
				}
			}
		} else {
			if (!name) {
				name = argv[i];
			} else {
				filterSize += strlen(argv[i]) + 1;
				filter = realloc(filter, filterSize);
				if (!filter) {
					perror("malloc()");
					exit(EXIT_FAILURE);
				}
				strcat(filter, argv[i]);
				strcat(filter, " ");
			}
		}
	}

	// if nids is to be used, check if it was compiled in
	if (stack == STACK_NIDS && !nidsAvailable())
		stack = STACK_INTERNAL;

	// perform task
	if (mode == MODE_AUTO || mode == MODE_FILE) {
		if (debug > 0) printf("Trying to read from file '%s'...\n", name);
		
		if (stack == STACK_INTERNAL) {
			result = sniffFile(name, filter);
		} else {
			result = nidsFile(name, filter);
		}
		
		if (result) {
			if (debug > 0) printf("Failed.\n");
			if (mode != MODE_AUTO) return (EXIT_FAILURE);
		} else {
			if (debug > 0) printf("Finished.\n");
			return (EXIT_SUCCESS);
		}
	}
		
	if (mode == MODE_AUTO || mode == MODE_DEV) {
		if (debug > 0) printf("Trying to read from device '%s'...\n", name);
		
		if (stack == STACK_INTERNAL) {
			result = sniffDevice(name, filter);
		} else {
			result = nidsDevice(name, filter);
		}
		
		if (result) {
			if (debug > 0) printf("Failed.\n");
			return (EXIT_FAILURE);
		} else {
			if (debug > 0) printf("Finished.\n");
			return (EXIT_SUCCESS);
		}
	}

	return (EXIT_FAILURE); // make compiler happy
}
