/*
 * HTTPSniff v0.2 - HTTP Response Sniffer
 * stream.c
 * (C)2004-2005 Michael Poppitz
 *
 * Processes data stream - and currently does the http parsing.
 *
 */
 
#include <stdio.h>
#include <string.h>

#include "http.h"
#include "stream.h"

FILE *streamCreateFile(char *extension) {
	FILE *f;
	char name[0x100];

	sprintf(name, "%.6u.%s", streamCount, extension);
	streamCount++;

	f = fopen(name, "w");
	if (f) {
		if (debug > 1) printf("streamCreateFile(): created %s\n", name);
	} else {
		perror("Could not open file for writing");
	}
	return (f);
}

u_int32_t streamWriteFile(FILE *f, u_char *data, u_int32_t size, u_int32_t outstanding) {
	u_int32_t written;
	u_int32_t min;

	min = size < outstanding ? size : outstanding;

	if ((written = fwrite(data, 1, min, f)) != min)
		perror("fwrite()");

	return (written);
}

int streamDetect(struct stream *s, int force) {
	u_char *start,*pos;
	int isHeader, isChunkHeader;

	if (s->used == 0) 
		return (0); // don't care for empty streams
	
	s->cache[s->used] = 0; // will stop string functions at end of used buffer the latest

	start = s->cache;
	if (s->type == HTTP_CHUNK && start[0] == '\r') {
		// chunked encoding has trailing crlf that must be stripped to get to next header
		start += 2; 
		if (s->used == 2) {
			s->used = 0;
			return (0);
		}
	}

	// determine if there is a complete header
	isHeader = httpIsHeader(start);
	isChunkHeader = httpIsChunkHeader(start);

	// if not existing/complete and processing is not forced, return
	if (!isHeader && !(s->type == HTTP_CHUNK && isChunkHeader) && !force)
		return (0);

	// unless its only a chunk header, the http type must be (re)detected
	if (s->type != HTTP_CHUNK || !isChunkHeader)
		s->type = httpType(start);

	if (s->type == HTTP_UNKNOWN) {
		s->file = streamCreateFile("raw");
		s->outstanding = 0xffffffff; /* FIX ME: outstanding should not be used for type unknown */
		if (debug > 1) printf("streamDetect(): detected HTTP_UNKNOWN\n");

	} else {
		// create new output file if needed
		if (!(s->type == HTTP_CHUNK && isChunkHeader) && s->type != HTTP_DISCARD) {
			if (s->file)
				fclose(s->file);
			s->file = streamCreateFile(httpFileType(start));
		}

		// determine expected/outstanding bytes
		switch (s->type) {
			case HTTP_SIMPLE:
				s->outstanding = 0xffffffff; /* FIX ME: outstanding should not be used */
				break;
			case HTTP_KEEP:
				s->outstanding = httpContentLength(start);
				break;
			case HTTP_CHUNK:
				s->outstanding = httpChunkLength(start);
				break;
			case HTTP_DISCARD:
				s->outstanding = 0;
				break;
			default:
				printf("streamDetect(): Internal error: Invalid type\n");
		}

		// realign cache
		pos = httpHeaderEnd(start);
		s->used -= pos - s->cache;
		memmove(s->cache, pos, s->used);

		if (debug > 1) {
			printf("streamDetect(): ");
			switch (s->type) {
				case HTTP_SIMPLE: printf("HTTP_SIMPLE"); break;
				case HTTP_KEEP:	printf("HTTP_KEEP with %u total bytes expected\n", s->outstanding); break;
				case HTTP_CHUNK: printf("HTTP_CHUNK"); break;
				case HTTP_DISCARD: printf("HTTP_DISCARD"); break;
			}
		}
	}
	return (1);
}

void streamOpen(struct stream *s) {
	s->file = NULL;
	s->used = 0;
	s->outstanding = 0;
	s->type = HTTP_NONE;
}

void streamWrite(struct stream *s, u_char *data, u_int32_t size) {
	size_t written;

	if (debug > 2) printf("streamWrite(): %u bytes received\n", size);
	
	// output file exists and is waiting for data
	if (s->file && s->outstanding > 0) {
		
		// first write leftover data from cache
		if (s->used > 0) {
			written = streamWriteFile(s->file, s->cache, s->used, s->outstanding);
			s->used -= written;
			s->outstanding -= written;
			if (s->used > 0)
				memmove(s->cache, s->cache + written, s->used);
			if (debug > 2) printf("streamWrite(): wrote %li bytes from cache, %u bytes left, %u bytes outstanding\n", written, s->used, s->outstanding);
		}

		// next write data from packet
		if (size > 0 && s->outstanding > 0) {
			written = streamWriteFile(s->file, data, size, s->outstanding);
			size -= written;
			s->outstanding -= written;
			data += written;
			if (debug > 2) printf("streamWrite(): wrote %li bytes from packet, %u bytes left, %u bytes outstanding\n", written, size, s->outstanding);
		}
	}
	
	// try to put leftovers in cache
	if (size != 0) {
		if (size < (HEADER_MAXSIZE - s->used)) {
			if (debug > 2) printf("  => adding all to cache\n");
			memcpy(s->cache + s->used, data, size);
			s->used += size;
			if (streamDetect(s, 0))
				streamWrite(s, NULL, 0);

		} else {
			if (debug > 2) printf("  => adding %u bytes to cache\n", HEADER_MAXSIZE - s->used);
			memcpy(s->cache + s->used, data, HEADER_MAXSIZE - s->used);
			data += HEADER_MAXSIZE - s->used;
			size -= HEADER_MAXSIZE - s->used;
			s->used = HEADER_MAXSIZE;

			// when cache is full, stream type must be determined
			streamDetect(s, 1);
			if (s->used < HEADER_MAXSIZE || s->outstanding > 0) {
				streamWrite(s, data, size);
			} else {
				printf("streamWrite(): OOPS: Cache full and no output can be written. Flushing cache...\n");
				s->used = 0;
				streamWrite(s, data, size);
			}
		}
	}
	
}

void streamClose(struct stream *s) {
	int i;

	if (debug > 2) printf("streamClose(): %u bytes in cache, %u bytes outstanding\n", s->used, s->outstanding);
	if (s->used) {
		// try to write what is left in cache, give up after max. 100 iterations
		for (i = 0; i < 100 && s->used > 0; i++) {
			streamDetect(s, 1);
			streamWrite(s, NULL, 0);
		}
		if (i == 100 && s->used > 0)
			printf("streamClose(): OOPS: Still content left after 100 rounds.\n");
	}
	if (s->file)
		fclose(s->file);
}

void streamInit() {
	streamCount = 0;
}
