/*
 * HTTPSniff v0.2 - HTTP Response Sniffer
 * sniff.c
 * (C)2004-2005 Michael Poppitz
 *
 * Communicates with libpcap and processes packet headers.
 *
 */

#include <string.h>
#include <net/ethernet.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>

#include <pcap.h>

#include "config.h"
#include "conn.h"
#include "sniff.h"

/***********************************************************************
 * TCP/IP Parser                                                       *
 ***********************************************************************/

void sniffTCP(u_char *packet, u_int32_t len, struct connid *cid) {
	struct tcphdr *tcp;
	int hdrsize;
	
	if (len < sizeof(struct tcphdr)) {
		if (debug > 5) printf("sniffTCP(): Packet too small for TCP header.\n");
		return;
	}

	tcp = (struct tcphdr *)packet;
	hdrsize = tcp->doff * 4;

	if (len < hdrsize) {
		if (debug > 5) printf("sniffTCP(): Packet too small for TCP header options.\n");
		return;
	}
	if (sizeof(struct tcphdr) > hdrsize) {
		if (debug > 5) printf("sniffTCP(): Invalid TCP header length field value.\n");
		return;
	}
		
	cid->tcpsrc = tcp->source;
	cid->tcpdst = tcp->dest;

	if (tcp->syn) {
		connAdd(cid, ntohl(tcp->seq));
	} else if (tcp->rst || tcp->fin) {
		connDelete(cid, ntohl(tcp->seq), packet + hdrsize, len - hdrsize);
	} else {
		connData(cid, ntohl(tcp->seq), packet + hdrsize, len - hdrsize);
	}
}

void sniffIP(u_char *packet, u_int32_t len) {
	struct connid cid;
	struct ip *ip;
	int frag;
	
	if (len < sizeof(struct ip)) {
		if (debug > 5) printf("sniffIP(): Packet too small for IP header.\n");
		return;
	}

	ip = (struct ip *)packet;

	frag = ntohs(ip->ip_off);
	if ((frag & IP_MF) || (frag & IP_OFFMASK)) {
		if (debug > 5) printf("sniffIP(): Fragmented IP not supported. (%i)\n", frag);
		return;
	}
	if (debug > 5) printf("sniffIP(): IP Protocol: %i\n", ip->ip_p);
	if (ip->ip_p == IPPROTO_TCP) {
		memcpy(&cid.ipsrc, &ip->ip_src, 4);
		memcpy(&cid.ipdst, &ip->ip_dst, 4);
		// if in doubt, go by total length field - some packets have trailing garbage
		if (ntohs(ip->ip_len) < len) {
			sniffTCP(packet + sizeof(struct ip), ntohs(ip->ip_len) - sizeof(struct ip), &cid);
		} else {
			sniffTCP(packet + sizeof(struct ip), len - sizeof(struct ip), &cid);
		}
	}
}

/***********************************************************************
 * Link Layer Parser                                                   *
 ***********************************************************************/

void sniffEther(u_char *packet, u_int32_t len) {
	struct ether_header *ether;

	if (len < sizeof(struct ether_header)) {
		if (debug > 5) printf("sniffEther(): Packet too small for ethernet header.\n");
		return;
	}
		
	ether = (struct ether_header *)packet;
	if (debug > 5) printf("sniffEther(): Ether Type: %i\n", ntohs(ether->ether_type));
	if (ntohs(ether->ether_type) == ETHERTYPE_IP) {
		sniffIP(packet + sizeof(struct ether_header), len - sizeof(struct ether_header));
	}
}

void sniffWLAN(u_char *packet, u_int32_t len) {
	// not much known about this one
	struct wlan {
		u_char stuff[30];
		u_int16_t ether_type;
	} *wlan;
	
	if (len < sizeof(struct wlan)) {
		if (debug > 5) printf("sniffWLAN(): Packet too small for WLAN header.\n");
		return;
	}
	
	wlan = (struct wlan *)packet;
	if (debug > 5) printf("sniffWLAN(): Ether Type: %i\n", ntohs(wlan->ether_type));
	if (ntohs(wlan->ether_type) == ETHERTYPE_IP) {
		sniffIP(packet + sizeof(struct wlan), len - sizeof(struct wlan));
	}
}

void sniffLinux(u_char *packet, u_int32_t len) {
	// this is a quick hack of the structure described in libpcap man page for linux cooked capture
	struct linux_sll {
		u_int16_t type;
		u_int16_t arphrd;
		u_int16_t llalen;
		u_char lladdr[8];
		u_int16_t ether_type;
	} *linuxsll;
	
	if (len < sizeof(struct linux_sll)) {
		if (debug > 5) printf("sniffLinux(): Packet too small for Linux cooked header.\n");
		return;
	}
	
	linuxsll = (struct linux_sll *)packet;
	if (debug > 5) printf("sniffLinux(): Ether Type: %i\n", ntohs(linuxsll->ether_type));
	if (ntohs(linuxsll->ether_type) == ETHERTYPE_IP) {
		sniffIP(packet + sizeof(struct linux_sll), len - sizeof(struct linux_sll));
	}
}

/***********************************************************************
 * libpcap Access                                                      *
 ***********************************************************************/

struct {
	int dlt;
	char *name;
	void (*handler)(u_char *, u_int32_t);
} dltTable[] = {
	{DLT_EN10MB, "EN10MB", sniffEther},
	{DLT_IEEE802_11, "IEEE802_11", sniffWLAN},
	{DLT_RAW, "RAW", sniffIP},
	{DLT_LINUX_SLL, "LINUX_SLL", sniffLinux},
	{0, "", NULL}
};

int sniffPcap(int type, char *res, char *filter) {
	void (*handler)(u_char *, u_int32_t);
	char errbuf[PCAP_ERRBUF_SIZE];
	pcap_t *pcap;
	struct pcap_pkthdr *hdr;
	struct bpf_program fp;
	u_char *packet;
	int dlt, i, status;
	
	/* depeding on type start offline or live capture */
	if (type) {
		pcap = pcap_open_offline(res, errbuf);
		if (!pcap) {
			printf("pcap_open_offline(): FATAL: %s\n", errbuf);
			return (-1);
		}
	} else {
		pcap = pcap_open_live(res, PACKET_MAXSIZE, 1, -1, errbuf);
		if (!pcap) {
			printf("pcap_open_live(): FATAL: %s\n", errbuf);
			return (-1);
		}
	}

	/* try to find handler for link type */
	dlt = pcap_datalink(pcap);
	handler = NULL;

	for (i = 0; dltTable[i].handler; i++) {
		if (dltTable[i].dlt == dlt) {
			handler = dltTable[i].handler;
			if (debug > 5) printf("Found data link layer type '%s'\n", dltTable[i].name);
		}
	}
	
	if (!handler) {
		printf("sniffPcap(): FATAL: Unknown data link layer. (%i)\n", dlt);
		pcap_close(pcap);
		return (-1);
	}

	/* if a filter was supplied, compile and activate it */
	if (filter) {
		if (pcap_compile(pcap, &fp, filter, 0, 0) == -1) {
			printf("pcap_compile(): FATAL: %s\n", pcap_geterr(pcap));
			return (-1);
		}

		if (pcap_setfilter(pcap, &fp) == -1) {
			printf("pcap_setfilter(): FATAL: %s\n", pcap_geterr(pcap));
			return (-1);
		}
	}

	/* process packets - abort only if end of save file reached */
	for (status = 0; status != -2; status = pcap_next_ex(pcap, &hdr, (const u_char **)&packet)) {
		if (debug > 5) printf("\n*** Read new packet of size %i ***\n", hdr->caplen);
		handler(packet, hdr->caplen);
	}

	/* clean up */
	if (filter) {
		pcap_freecode(&fp);
	}

	pcap_close(pcap);

	return (0);
}

void sniffInit() {
	connInit();
}

void sniffKill() {
	connKill();
}

int sniffDevice(char *dev, char *filter) {
	return (sniffPcap(0, dev, filter));
}

int sniffFile(char *file, char *filter) {
	return (sniffPcap(1, file, filter));
}
