/*
 * HTTPSniff v0.2 - HTTP Response Sniffer
 * hash.c
 * (C)2004-2005 Michael Poppitz
 *
 * Maintains hash table for fast connection lookups.
 *
 */

#include <stdio.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "config.h"
#include "hash.h"

struct connection *hashTable[CONNECTION_BUCKETS];
int hashTableEntries;

/* debugging function - outputs hash table contents */
void hashDebug() {
	struct connection *tmp;
	char source[16];
	int i;
		
	for (i = 0; i < CONNECTION_BUCKETS; i++) {
		printf("%2i:", i);
		for (tmp = hashTable[i]; tmp; tmp = tmp->next) {
			
			strncpy(source, inet_ntoa(tmp->cid.ipsrc), 15);
			printf(
				"(%s:%i<->%s:%i) ",
				source,
				ntohs(tmp->cid.tcpsrc),
				inet_ntoa(tmp->cid.ipdst),
				ntohs(tmp->cid.tcpdst)
			);	
		
		}
		printf("\n");
	}
}

/* helper function - calculates hash of a connid */
u_int hashCalc(struct connid *cid) {
	int i;
	u_int hash;

	/* no need to do elaborated hashing - port numbers and ips are random enough*/
	hash = 0;
	for (i = 0; i < sizeof(struct connid); i++) {
		hash += ((u_char *)cid)[i];
	}
	return (hash % CONNECTION_BUCKETS);
}

/* initializes hash buckets */
void hashInit() {
	int i;
	
	for (i = 0; i < CONNECTION_BUCKETS; i++)
		hashTable[i] = NULL;

	hashTableEntries = 0;
}

/* finds connection matching given connection id */
struct connection *hashFind(struct connid *cid) {
	struct connection *tmp;
	
	tmp = hashTable[hashCalc(cid)];
	while (tmp) {
		if (memcmp(&tmp->cid, cid, sizeof(struct connid)) == 0) {
			return (tmp);
		}
		tmp = tmp->next;
	}
	return (NULL);
}

/* finds first connection in hash table */
struct connection *hashFindAll() {
	int i;
	
	for (i = 0; i < CONNECTION_BUCKETS; i++)
		if (hashTable[i])
			return hashTable[i];
	return (NULL);
}

int hashAdd(struct connection *conn) {
	struct connection *tmp;
	int bucket;

	if (hashTableEntries >= MAX_CONNECTIONS)
		return (0);

	conn->next = NULL;
	bucket = hashCalc(&conn->cid);
	
	if(!hashTable[bucket]) {
		hashTable[bucket] = conn;
	} else {
		for (tmp = hashTable[bucket]; tmp->next; tmp = tmp->next);
		tmp->next = conn;
	}

	hashTableEntries++;
	return (1);
}

struct connection *hashDelete(struct connid *cid) {
	struct connection *tmp, *ret;
	int bucket;

	bucket = hashCalc(cid);
	tmp = hashTable[bucket];
	
	if (!tmp)
		return (NULL);
	
	if (memcmp(&tmp->cid, cid, sizeof(struct connid)) == 0) {
		hashTable[bucket] = tmp->next;
		hashTableEntries--;
		return (tmp);
	
	} else {
		while (tmp->next && (memcmp(&tmp->next->cid, cid, sizeof(struct connid)) != 0)) {
			tmp = tmp->next;
		}
		if (tmp->next) {
			ret = tmp->next;
			tmp->next = tmp->next->next;
			hashTableEntries--;
			return (ret);
		}
	}

	return (NULL);
}
