/*
 * HTTPSniff v0.2 - HTTP Response Sniffer
 * conn.c
 * (C)2004-2005 Michael Poppitz
 *
 * Assigns individual packets to connections and takes care of proper packet order.
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "config.h"
#include "hash.h"
#include "segment.h"
#include "stream.h"
#include "conn.h"

struct connection connHashTable[CONNECTION_BUCKETS];

/* helper function for debugging output */
void connDebug(struct connid *cid, char *mesg) {
	char source[16];

	strncpy(source, inet_ntoa(cid->ipsrc), 15);
	printf(
		"(%s:%i<->%s:%i) -- %s\n",
		source,
		ntohs(cid->tcpsrc),
		inet_ntoa(cid->ipdst),
		ntohs(cid->tcpdst),
		mesg
	);	
}

/* gets cid of current connections twin */
void connTwin(struct connid *twin, struct connid *cid) {
	memcpy(&twin->ipsrc, &cid->ipdst, sizeof(struct in_addr));
	memcpy(&twin->ipdst, &cid->ipsrc, sizeof(struct in_addr));
	twin->tcpsrc = cid->tcpdst;
	twin->tcpdst = cid->tcpsrc;
}

/* add connection */
void connAdd(struct connid *cid, u_int32_t seq) {
	struct connection *conn;
	
	conn = hashFind(cid);
	if (!conn) {
		conn = (struct connection *)malloc(sizeof(struct connection));
		if (!conn) {
			perror("connAdd(): Cannot add connection");
			return;
		}
		memcpy(&conn->cid, cid, sizeof(struct connid));
		conn->seq = seq + 1;
		conn->segments = NULL;
		conn->time = time(NULL);
		conn->size = 0;
		streamOpen(&conn->stream);
		if(!hashAdd(conn))
			printf("connAdd(): OOPS: Connection limit exceeded.\n");
		if (debug > 4) connDebug(cid, "Added new connection.");
	} else {
		conn->time = time(NULL);
		if (debug > 4) connDebug(cid, "Ignored add. Connection already exists.");
	}
}

void connDeleteInternal(struct connection *conn) {
	hashDelete(&conn->cid);
	segmentFree(&conn->segments);
	streamClose(&conn->stream);
	free(conn);
}

void connDataInternal(struct connid *cid, u_int32_t seq, u_char *packet, u_int32_t len, u_char flags) {
	struct connection *conn;
	struct segment *seg;
		
	conn = hashFind(cid);
	if (!conn) {
		if (debug > 4) connDebug(cid, "Whoops. Data for uninitialized connection. Adding...");
		connAdd(cid, seq + len);
		conn = hashFind(cid);
	}
	
	if (debug > 4) connDebug(cid, "Processing data...");
	if (debug > 3) printf("   => segment add: %u bytes (seq: %u) (flags: %i)\n", len, seq, flags);
	segmentAdd(&conn->segments, seq, packet, len, flags);
	if (conn->seq == seq) {
		while (conn && (seg = segmentFind(conn->segments, conn->seq))) {
			if (seg->size > 0) {
				conn->seq += seg->size;
				conn->size += seg->size;
				if (debug > 3) printf("   => segment match: %u bytes (%u total) (new seq: %u)\n", len, conn->size, conn->seq);
				streamWrite(&conn->stream, seg->data, seg->size);
			}
			if (seg->flags & SEGMENT_FLAG_CLOSE) {
				if (debug > 3) {
					printf("   => close flag encountered: closing connection\n");
					segmentDump(conn->segments);
				}
				connDeleteInternal(conn);
				conn = NULL;
			} else {
				segmentDelete(&conn->segments, seg->seq);
			}
		}
	}
}

/* delete connection */
void connDelete(struct connid *cid, u_int32_t seq, u_char *data, u_int32_t len) {
	connDataInternal(cid, seq, data, len, SEGMENT_FLAG_CLOSE);
}

/* add data to a connection */
void connData(struct connid *cid, u_int32_t seq, u_char *data, u_int32_t len) {
	connDataInternal(cid, seq, data, len, 0);
}

/* called on startup */
void connInit() {
	hashInit();
	streamInit();
}

/* called on shutdown */
void connKill() {
	struct connection *conn;
	
	// this is not very efficient - but only needed once
	while ((conn = hashFindAll())) {
		connDeleteInternal(conn);
	}
}
