/*
 * HTTPSniff v0.1 - HTTP Response Sniffer
 * stream.c
 * (C)2004 Michael Poppitz
 *
 * Processes data stream - and currently does the http parsing.
 *
 */
 
#include <stdio.h>
#include <string.h>

#include "stream.h"

// FIX ME: all the http stuff needs to go someplace else
int streamType(u_char *hdr) {
	u_char *end, *pos;

	// first try to eliminate non http streams
	if(strncmp(hdr, "HTTP", 4))
		return (STREAM_UNKNOWN);

	end = strstr(hdr, "\r\n\r\n");
	if (!end) return (STREAM_UNKNOWN);

	// pos = strstr(hdr, "Content-Type:");
	// if (!pos || pos > end)
	//	return (STREAM_UNKNOWN);

	// now try to distinguish different http types
	pos = strstr(hdr, "Transfer-Encoding: chunked");
	if (pos && pos < end)
		return (STREAM_CHUNK);

	pos = strstr(hdr, "Connection: Keep-Alive");
	if (pos && pos < end)
		return (STREAM_KEEP);
	
	return (STREAM_SIMPLE);
}

char streamContentTypeBuffer[11];
char *streamContentType(u_char *hdr) {
	char *unknown = "unknown";
	u_char *pos, *end;
	int i;
	
	end = strstr(hdr, "\r\n\r\n");
	pos = strstr(hdr, "Content-Type: ");

	if (pos && pos < end) { 
		pos = strchr(pos + 14, '/') + 1;
		for (i = 0; i < 10; i++) {
			streamContentTypeBuffer[i] = pos[i];
		
			if (
				streamContentTypeBuffer[i] == '\r'
				|| streamContentTypeBuffer[i] == ';'
				|| streamContentTypeBuffer[i] == ' '
			)
				streamContentTypeBuffer[i] = 0;
			if (streamContentTypeBuffer[i] == 0)
				return (streamContentTypeBuffer);
		}
		streamContentTypeBuffer[i] = 0;
	} else {
		return (unknown);
	}
	
	return (streamContentTypeBuffer);
}

u_int32_t streamContentLength(u_char *hdr) {
	u_char *pos, *end;
	u_int32_t len;
	
	len = 0;
	end = strstr(hdr, "\r\n\r\n");
	pos = strstr(hdr, "Content-Length: ");
	if (pos && pos < end)
		if (pos[16] >= '0' || pos[16] <= '9')
			sscanf(pos + 16, "%u", &len);
	
	return (len);
}

FILE *streamCreateFile(char *extension) {
	FILE *f;
	char name[0x100];

	sprintf(name, "%.6u.%s", streamCount, extension);
	streamCount++;

	f = fopen(name, "w");
	if (f) {
		if (debug > 1) printf("streamCreateFile(): created %s\n", name);
	} else {
		perror("Could not open file for writing");
	}
	return (f);
}

void streamFlush(struct stream *s) {
	u_char *pos;
	size_t size;

	if (s->used == 0) 
		return; // don't care for empty streams
	
	s->cache[s->used] = 0; // will stop string functions at end of used buffer the latest
	s->type = streamType(s->cache);

	if (s->type == STREAM_KEEP) {
		s->outstanding = streamContentLength(s->cache);
		if (debug > 1) printf("streamFlush(): detected STREAM_KEEP with %u total bytes expected\n", s->outstanding);
		
		
		pos = strstr(s->cache, "\r\n\r\n") + 4;
		size = s->used - (pos - s->cache);
		if (size >= s->outstanding) {
			if (s->outstanding > 0) {
				s->file = streamCreateFile(streamContentType(s->cache));
				if (fwrite(pos, 1, s->outstanding, s->file) != s->outstanding)
					perror("fwrite()");
				if (debug > 2) printf("streamFlush(): wrote %u bytes\n", s->outstanding);
				fclose(s->file);
			}
			s->file = NULL;
			pos += s->outstanding;
			s->used = size - s->outstanding;
			s->outstanding = 0;
			if (debug > 2) printf("streamFlush(): %u bytes left in cache\n", s->used);
			if (s->used > 0)
				memmove(s->cache, pos, s->used);
		} else {
			s->file = streamCreateFile(streamContentType(s->cache));
			s->outstanding -= size;
			s->used = 0;
		}
	
	} else if (s->type == STREAM_SIMPLE) {
		s->file = streamCreateFile(streamContentType(s->cache));
		s->outstanding = 0xffffffff; /* FIX ME: outstanding should not be used for type simple */
		if (debug > 1) printf("streamFlush(): detected STREAM_SIMPLE\n");

		pos = strstr(s->cache, "\r\n\r\n") + 4;
		size = s->used - (pos - s->cache);
		s->used = 0;
	
	} else {
		s->file = streamCreateFile("raw");
		s->outstanding = 0xffffffff; /* FIX ME: outstanding should not be used for type unknown */
		if (debug > 1) printf("streamFlush(): detected STREAM_UNKNOWN\n");

		pos = s->cache;
		size = s->used;
		s->used = 0;
	}
	
	if (s->file) {
		if (fwrite(pos, 1, size, s->file) != size)
			perror("fwrite()");
		if (debug > 2) printf("streamFlush(): wrote %lu bytes\n", size);
	}
}

void streamOpen(struct stream *s) {
	s->file = NULL;
	s->used = 0;
	s->outstanding = 0;
	s->type = STREAM_UNKNOWN;
}

void streamWrite(struct stream *s, u_char *data, u_int32_t size) {
	if (debug > 2) printf("streamData(): %u bytes received, %u bytes outstanding\n", size, s->outstanding);
	
 	if (s->file) {
		if (s->outstanding >= size) {
			if (fwrite(data, 1, size, s->file) != size)
					perror("fwrite()");
			s->outstanding -= size;
			data += size;
			size = 0;
		} else {
			fwrite(data, 1, s->outstanding, s->file);
			data += s->outstanding;
			size -= s->outstanding;
			s->outstanding = 0;
		}
		if (s->outstanding == 0) {
			fclose(s->file);
			s->file = NULL;
		}
	}
	
	if (!s->file && size != 0) {
		if (size < (HEADER_MAXSIZE - s->used)) {
			if (debug > 2) printf("  => adding all to cache\n");
			memcpy(s->cache + s->used, data, size);
			s->used += size;
		} else {
			if (debug > 2) printf("  => adding %u bytes to cache\n", HEADER_MAXSIZE - s->used);
			memcpy(s->cache + s->used, data, HEADER_MAXSIZE - s->used);
			data += HEADER_MAXSIZE - s->used;
			size -= HEADER_MAXSIZE - s->used;
			s->used = HEADER_MAXSIZE;
			streamFlush(s);
			streamWrite(s, data, size);
		}
	}
	
}

void streamClose(struct stream *s) {
	if (debug > 2) printf("streamClose(): %u bytes in cache, %u bytes outstanding\n", s->used, s->outstanding);
	streamFlush(s);
	if (s->file)
		fclose(s->file);
}

void streamInit() {
	streamCount = 0;
}
