/*
 * HTTPSniff v0.1 - HTTP Response Sniffer
 * conn.c
 * (C)2004 Michael Poppitz
 *
 * Assigns individual packets to connections and takes care of proper packet order.
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "config.h"
#include "hash.h"
#include "frag.h"
#include "stream.h"
#include "conn.h"

struct connection connHashTable[CONNECTION_BUCKETS];

/* helper function for debugging output */
void connDebug(struct connid *cid, char *mesg) {
	char source[16];

	strncpy(source, inet_ntoa(cid->ipsrc), 15);
	printf(
		"(%s:%i<->%s:%i) -- %s\n",
		source,
		ntohs(cid->tcpsrc),
		inet_ntoa(cid->ipdst),
		ntohs(cid->tcpdst),
		mesg
	);	
}

/* gets cid of current connections twin */
void connTwin(struct connid *twin, struct connid *cid) {
	memcpy(&twin->ipsrc, &cid->ipdst, sizeof(struct in_addr));
	memcpy(&twin->ipdst, &cid->ipsrc, sizeof(struct in_addr));
	twin->tcpsrc = cid->tcpdst;
	twin->tcpdst = cid->tcpsrc;
}

/* initialize connection */
void connAdd(struct connid *cid, u_int32_t seq) {
	struct connection *conn;
	
	conn = hashFind(cid);
	if (!conn) {
		conn = (struct connection *)malloc(sizeof(struct connection));
		if (!conn) {
			perror("connAdd(): Cannot add connection");
			return;
		}
		memcpy(&conn->cid, cid, sizeof(struct connid));
		conn->seq = seq + 1;
		conn->frags = NULL;
		conn->time = time(NULL);
		conn->size = 0;
		streamOpen(&conn->stream);
		hashAdd(conn);
		if (debug > 4) connDebug(cid, "Added new connection.");
	} else {
		conn->time = time(NULL);
		if (debug > 4) connDebug(cid, "Ignored add. Connection already exists.");
	}
}

/* kill connection */
void connDelete(struct connid *cid, u_int32_t seq) {
	struct connection *conn;
	
	conn = hashDelete(cid);
	if (conn) {
		streamClose(&conn->stream);
		free(conn);
		if (debug > 4) connDebug(cid, "Deleted connection.");
	} else {
		if (debug > 4) connDebug(cid, "Ignored delete. Connection does not exist.");
	}
}

/* add data to a connection */
void connData(struct connid *cid, u_int32_t seq, u_char *packet, u_int32_t len) {
	struct connection *conn;
	struct fragment *frag;
		
	conn = hashFind(cid);
	if (len == 0) {
		if (conn) {
			conn->time = time(NULL);
			if (debug > 4) connDebug(cid, "No data in packet.");
		} else {
			connAdd(cid, seq);
		}
		return;
	}
	
	if (!conn) {
		if (debug > 4) connDebug(cid, "Whoops. Data for uninitialized connection. Adding...");
		connAdd(cid, seq + len);
		conn = hashFind(cid);
	}
	
	if (debug > 3) printf("base seq %u, packet seq %u\n", conn->seq, seq);
	if (conn->seq == seq) {
		conn->seq += len;
		conn->size += len;
		streamWrite(&conn->stream, packet, len);
		if (debug > 3) printf("   => direct match: %u bytes (%u total) (new seq: %u)\n", len, conn->size, conn->seq);

		while ((frag = fragFind(conn, conn->seq))) {
			conn->seq += frag->size;
			conn->size += frag->size;
			streamWrite(&conn->stream, frag->data, frag->size);
			if (debug > 3) printf("   => fragment match: %u bytes (%u total) (new seq: %u)\n", len, conn->size, conn->seq);
		}
		fragDelete(conn, conn->seq);
	} else {
		fragAdd(conn, seq, packet, len);
		if (debug > 3) printf("   => fragment saved: %u bytes (%u total) (new seq: %u)\n", len, conn->size, conn->seq);
	}
	
	if (debug > 4) connDebug(cid, "Added data to connection.");
}

/* called on startup */
void connInit() {
	hashInit();
	streamInit();
}

/* called on shutdown */
void connKill() {
	struct connection *conn;
	
	// this is not very efficient - but only needed once
	while ((conn = hashFindAll())) {
		connDelete(&conn->cid, conn->seq);
	}
}
