; ======================================================================
; Frequency Counter Software
;
; show.asm
;
; Copyright (C) 2005 Michael Poppitz
;
; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or (at
; your option) any later version.
;
; This program is distributed in the hope that it will be useful, but
; WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
; General Public License for more details.
;
; You should have received a copy of the GNU General Public License along
; with this program; if not, write to the Free Software Foundation, Inc.,
; 51 Franklin St, Fifth Floor, Boston, MA 02110, USA
;
; ======================================================================
; For details see: http://www.sump.org/projects/counter/
;
; Display procedures
; ======================================================================
.cseg

;===== displays the 32+8 float given in fr =============================

show:
	clr	c			; comma positioned outside display
	ldi	a, 32
	cp	fre, a			; exponent 32 indicates integer
	breq	show_do			; that requires no scaling

	ldi	zl, low(2 * f0)		; select start of scaling table
	ldi	zh, high(2 * f0)
	ldi	c, 8			; comma position

show_next:
	clr	fa0
	clr	fa1
	lpm	fa2, Z+
	lpm	fa3, Z+
	lpm	fae, Z+

	cp	fre, fae		; is value's exponent lower?
	brlo	show_scale		; if so, scaling entry applies
	brne	show_bigger		; if bigger, dont compare value
	cp	fr3, fa3
	brlo	show_scale
	brne	show_bigger
	cp	fr2, fa2
	brlo	show_scale

show_bigger:
	adiw	zl, 3			; select next entry
	dec	c			; increase comma position
	cpi	zl, low(2 * fend)	; end reached? (works for <=32 entries)
	brne	show_next		; if not, try again
	clr	c			; otherwise position comma outside
	rjmp	show_conv		; and simply convert to integer

show_scale:
	lpm	fa0, Z+			; load scaling factor to a
	lpm	fa1, Z+
	lpm	fa2, Z+
	clr	fa3
	ldi	a, 32
	mov	fae, a

	mov	fb0, fr0		; load result to b
	mov	fb1, fr1
	mov	fb2, fr2
	mov	fb3, fr3
	mov	fbe, fre

	rcall	mul32f8			; scale
show_conv:
	rcall	conv32f8toi		; convert to integer
show_do:
	mov	fre, c			; abuse former exponent for comma pos.
	rcall	show_int		; display

	ret

;===== convert fr to decimal and output to display buffer ==============

show_int:
	ldi	zh, high(2 * d7)
	ldi	zl, low(2 * d7)		; load table with decimal constants

	tst	fre			; display comma?
	breq	show_int_no_comma	; if no, ok
	adiw	zl, 4			; if yes, display as 1/10th capacity
show_int_no_comma:

	ldi	xh, high(DISPLAY + 8)
	ldi	xl, low(DISPLAY + 8)	; load end of display buffer
	ldi	i, 8			; maximum size 8 digits

	ldi	a, 6
	cp	fr3, a			; simple limit exceeded check
	brsh	show_int_exceeded	; if not exceeded, display number


	ldi	f, 0x0f			; indicates leading blank digits

show_int_next_digit:
	cp	i, fre			; check if comma position
	breq	show_int_comma

	clr	e			; clear current digit value
	lpm	a, Z+			; load highest decimal byte
	lpm	b, Z+
	lpm	c, Z+
	lpm	d, Z+			; load lowest decimal byte

show_int_next_sub:
	sub	fr0, a			; substract decimal constant
	sbc	fr1, b
	sbc	fr2, c
	sbc	fr3, d

	brlo	show_int_too_big	; if carry, count was lower

	inc	e			; otherwise increase digit value
	jmp	show_int_next_sub	; and continue subtracting

show_int_too_big:
	add	fr0, a
	adc	fr1, b			; restore last positive value
	adc	fr2, c			; smaller then decimal constant
	adc	fr3, d			; for next round

	cpi	e, 10			; is digit value >= 10?
	brsh	show_int_exceeded	; if so, limit must be exceeded

	ld	a, -X			; load previous byte from buffer
	andi	a, 0xf0			; keep only position bits
	andi	e, 0x0f			; protect display buffer
	or	a, e			; add new value

	cpi	d, 0x01			; is last digit?
	breq	show_int_anyway		; if so, always display

	tst	e			; is digit zero?
	breq	show_int_zero
	clr	f			; if not clear blanking flag
show_int_zero:
	or	a, f			; or in blanking flag

show_int_anyway:
	st	X, a			; write back

	dec	i			; decrease loop counter
	brne	show_int_next_digit	; and continue if not zero
	jmp	show_int_end

	;===== output comma sign

show_int_comma:
	ld	a, -X
	andi	a, 0xf0			; keep only position bits
	ori	a, 0x0a			; add 'c' sign
	st	X, a
	clr	f			; clear blanking flag

	dec	i
	brne	show_int_next_digit
	jmp	show_int_end

	;===== output limit exceeded

show_int_exceeded:
	ldi	xh, high(DISPLAY + 8)	; should still be set to this
	ldi	xl, low(DISPLAY + 8)	; when limit exceeded, but be paranoid

	ldi	b, 0x0c			; c will be displayed as "u"
show_int_ex_next:			; so the loop will output "uuuuuuuu"
	ld	a, -X			; as poor man's error indication
	andi	a, 0xf0
	or	a, b
	st	X, a
	dec	i
	brne	show_int_ex_next

show_int_end:
	ret

;===== initialize display buffer =======================================

show_splash:
	ldi	zh, high(2 * splash)	; select "splash screen" data
	ldi	zl, low(2 * splash)
	ldi	xh, high(DISPLAY)	; select display buffer
	ldi	xl, low(DISPLAY)
	ldi	i, 8			; copy 8 bytes

show_splash_next:
	lpm	a, Z+
	st	X+, a
	dec	i
	brne	show_splash_next

	ret

;===== constants =======================================================

d7:	.db	0x80, 0x96, 0x98, 0x00	; 10.000.000
d6:	.db	0x40, 0x42, 0x0F, 0x00	; 1.000.000
d5:	.db	0xA0, 0x86, 0x01, 0x00	; 100.000
d4:	.db	0x10, 0x27, 0x00, 0x00	; 10.000
d3:	.db	0xE8, 0x03, 0x00, 0x00	; 1.000
d2:	.db	0x64, 0x00, 0x00, 0x00	; 100
d1:	.db	0x0A, 0x00, 0x00, 0x00	; 10
d0:	.db	0x01, 0x00, 0x00, 0x00	; 1

f0:	.db	0x00, 0x80		; < 0.5
	.db	0x01, 0x80, 0x96, 0x98	; * 2^1, scale by 10.000.000
f1:	.db	0x00, 0xA0		; < 0.625
	.db	0x04, 0x40, 0x42, 0x0F	; * 2^4, scale by 1.000.000
f2:	.db	0x00, 0xC8		; < 0.78125
	.db	0x07, 0xA0, 0x86, 0x01	; * 2^7, scale by 100.000
f3:	.db	0x00, 0xFA		; < 0.9765625
	.db	0x0A, 0x10, 0x27, 0x00	; * 2^10, scale by 10.000
f4:	.db	0x40, 0x9C		; < 0.6103515625
	.db	0x0E, 0xEA, 0x03, 0x00	; * 2^14, scale by 1.000
f5:	.db	0x50, 0xC3		; < 0.762939453125
	.db	0x11, 0x00, 0x00, 0x64	; * 2^17, scale by 100
f6:	.db	0x24, 0xF4		; < 0.95367431640625
	.db	0x14, 0x00, 0x00, 0x0A	; * 2^20, scale by 10
fend:

splash:	.db	0x08, 0x18, 0x48, 0x58	; will be displayed for
	.db	0x88, 0x98, 0xc8, 0xd8	; the first interval on power up

;===== variables =======================================================

.dseg

DISPLAY:	.byte 8			; display buffer
DISPLAYEND:
