; ======================================================================
; Frequency Counter Software
;
; main.asm
;
; Copyright (C) 2005 Michael Poppitz
;
; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or (at
; your option) any later version.
;
; This program is distributed in the hope that it will be useful, but
; WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
; General Public License for more details.
;
; You should have received a copy of the GNU General Public License along
; with this program; if not, write to the Free Software Foundation, Inc.,
; 51 Franklin St, Fifth Floor, Boston, MA 02110, USA
;
; ======================================================================
; For details see: http://www.sump.org/projects/counter/
;
; Contains timer mode specific code
; ======================================================================

;=======================================================================
;
; Frequency Counter Software
;
;
; Copyright by Michael "Mr. Sump" Poppitz.
;
;=======================================================================
;
; Pin configuration:
;	PD0-PD3		bcd output
;	PD4,PD6,PD7	digit number output 
;	PD5(T1)		counter signal input
;	PB1-PB4		external prescaler high nibble input
;	PC0-PC3		external prescaler low nibble input
;	PC4		counter signal latch enable output
;
;=======================================================================

.include "inc/iom16.h"
.include "defs.h"

.cseg

;===== interrupt vectors ===============================================

	rjmp	init			; reset
	reti				; int0
	reti				; int1
	reti				; timer2 compare
	reti				; timer2 overflow
	rjmp	read			; timer1 capture
	reti				; timer1 compare a
	reti				; timer1 compare b
	rjmp	count			; timer1 overflow
	rjmp	scheduler		; timer0 overflow

;===== include functions ===============================================

.include "counter.asm"			; these go here because
.include "timer.asm"			; avra ignores .org
.include "math.asm"
.include "show.asm"

.cseg

;===== initialize ======================================================

init:	cli				; disable all interrupts

	ldi	a, high(RAMEND)		; set stack pointer to end of RAM
	out	sph, a
	ldi	a, low(RAMEND)
	out	spl, a

	;===== configure i/o pins

	ldi	a, 0b11011111		; set port D to output except T1
	out	DDRD, a

	ldi	a, 0b11110000		; lowest four bits input, rest output
	out	DDRC, a			; for port C
	clr	a			; no pull up
	out	PORTC, a		; for port C
	ldi	a, 0b01100000		; 5,6 output, rest input
	out	DDRB, a			; for port B
	ldi	a, 0b10000000		; pb7 with pull up, rest without
	out	PORTB, a		; for port B

	;===== configure timer0

	ldi	a, 0b00000011		; CLK/64
	out	TCCR0, a		; select timer0 prescaling

	ldi	a, 0b00000001		; enable timer0 overflow int
	out	TIMSK, a

	;===== show splash screen

	rcall	show_splash

	;===== configure watchdog timer

	ldi	a, 0b00001000		; watchdog enable, timeout 16ms
	out	WDTCR, a		; write watchdog configuration

	;===== initialize registers

	clr	io_pos
	clr	count_val
	clr	scale
	ldi	flags, 0		; load default flags
	sts	lastFlags, flags

	;===== start operation

	rcall	init_count		; initialize counter mode

	sei				; enable interrupts
	sbi	PORTC, 4		; activate external signal

	rjmp	run			; done, go to main program

;===== main ============================================================

run:	sleep
	rjmp	run

;===== timer capture handler ===========================================

read:	rcall	read_timer		; call read function
	reti

;===== counter overflow handler ========================================

count:	inc	count_val		; increase software counter
	reti

;===== scheduler (sort of :-) ==========================================

scheduler:
	ldi	a, t0init		; preset counter
	out	TCNT0, a		; for next overflow in 1/1000s
	inc	scale			; increase round counter

	rcall	io			; call to io task due every round

	cpi	scale, 1000 / rate	; measure point reached?
	brne	scheduler_end

	clr	scale

	lds	a, lastFlags

	mov	b, a
	andi	b, F_TIMER
	breq	scheduler_notimer

	rcall	calc_timer		; run data display
	rjmp	scheduler_end

scheduler_notimer:
	mov	b, a
	andi	b, F_COUNT
	breq	scheduler_nocount

	rcall	read_count		; run timer readout

	in	a, SFIOR		; reset prescaler 
	ori	a, 0b00000001		; (external counter was stopped)
	out	SFIOR, a

	ldi	a, t0init		; preset counter
	out	TCNT0, a		; for next overflow in 1/1000s

	rcall	calc_count		; run data display

scheduler_nocount:

scheduler_end:
	reti

;===== cycle displayed digit & read switch =============================

io:	;===== read switch setting

	in	d, PINB			; read switch bit
	mov	c, io_pos		; rescue current switch id

	;===== display digit

	inc	io_pos
	mov	a, io_pos
	andi	a, 0x07			; lowest 3 bits mark buffer pos.

	ldi	xh, high(DISPLAY)
	ldi	xl, low(DISPLAY)	; load display buffer
	add	xl, a			; calculate position in RAM
	ld	b, X			; fetch byte
	out	PORTD, b		; write to display hardware
	wdr				; clear watchdog timer

	;===== process switch setting

	andi	c, 0x07
	inc	c
	mov	a, c

io_rotout:
	ror	flags			; move selected bit into carry
	dec	c
	brne	io_rotout

	cpi	d, 0x80			; will set carry if PB7 is 0, and switch on

io_rotback:
	rol	flags			; move flags back to original pos.
	dec	a
	brne	io_rotback

	;===== handle flags after each round

	mov	a, io_pos
	andi	a, 0x07
	brne	io_end

	lds	a, lastFlags
	eor	a, flags		; obtain bitmask of changed flags

	andi	a, F_COUNT | F_TIMER	; has mode changed
	breq	io_end_round		; if not, end

	mov	b, flags
	andi	b, F_COUNT | F_TIMER
	cpi	b, F_TIMER
	brne	io_notimer

	rcall	init_timer
	rjmp	io_end_round

io_notimer:
	cpi	b, F_COUNT
	brne	io_end_round

	rcall	init_count

io_end_round:
	sts	lastFlags, flags
io_end:
	ret


;===== variables =======================================================

.dseg

lastFlags:	.byte	1		; last round's flag byte

;===== eof =============================================================
