; ======================================================================
; Frequency Counter Software
;
; counter.asm
;
; Copyright (C) 2005 Michael Poppitz
;
; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or (at
; your option) any later version.
;
; This program is distributed in the hope that it will be useful, but
; WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
; General Public License for more details.
;
; You should have received a copy of the GNU General Public License along
; with this program; if not, write to the Free Software Foundation, Inc.,
; 51 Franklin St, Fifth Floor, Boston, MA 02110, USA
;
; ======================================================================
; For details see: http://www.sump.org/projects/counter/
;
; Contains counter mode specific code
; ======================================================================
.cseg

;===== initialisation for counter mode =================================

init_count:
	ldi	yh, high(CNTBUFEND)	; select counter buffer end
	ldi	yl, low(CNTBUFEND)

	clr	a
	ldi	i, rate * 4		; $rate entries, 4 bytes each
init_cntbuf:
	st	-Y, a			; clear byte
	dec	i
	brne	init_cntbuf		; if not done, do next
 					; Y now points to buffer start

	ldi	a, 0b00000000		; normal operation
	out	TCCR1A, a
	ldi	a, 0b00000110		; ext source t1, falling edge
	out	TCCR1B, a

	ldi	a, 0b00000101		; enable timer0 & 1 overflow int
	out	TIMSK, a

	ldi	a, 0b11111101		; clear pending interrupts
	out	TIFR, a

	clr	count_val
	clr	a
	out	TCNT1H, a
	out	TCNT1L, a		; clear internal counter

	ret

;===== read counter value ==============================================

read_count:
	cbi	PORTC, 4		; stop external counter

	ldi	c, 4
read_count_settle:
	nop				; let external counter settle
	dec	c
	brne	read_count_settle

	in	a, PINB			; read value of ext. 8bit counter
	in	b, PINC			;
	lsr	a			; make port n pin 1-4 bit 0-3 
	andi	a, 0x0f			; take the lower nibble of port b
	swap	a			; make it high nibble
	andi	b, 0x0f			; take the lower nibble of portc
	or	a, b			; combine both to byte
	mov	fr0, a			; and write to lowest counter byte

	in	fr1, TCNT1L		; read value of internal 16bit counter
	in	fr2, TCNT1H

	in	a, TIFR			; fetch counter interrupt flags
	sbrc	a, 2			; did int. counter overflow meanwhile?
	inc	count_val		; if so increase software counter
	andi	a, 0b00000100		; mask out all but t1 overflow
	out	TIFR, a			; and clear flag if set

	mov	fr3, count_val		; read value of 8bit software counter

	sbi	PORTC, 4		; start external counter again

	ldi	i, calib		; modify the delay between counter
read_count_cal:				; and timer activation for calibration
	dec	i
	brne	read_count_cal

	reti

;===== calculate and output frequency ==================================

calc_count:
	mov	xh, yh			; make copy of Y in X
	mov	xl, yl

	ld	d, X+			; load oldest counter value 
	ld	c, X+
	ld	b, X+
	ld	a, X+

	st	Y+, fr0		; store current counter value
	st	Y+, fr1
	st	Y+, fr2
	st	Y+, fr3

	cpi	yh, high(CNTBUFEND)	; check if end of buffer reached
	brlo	calc_count_nobufend
	brne	calc_count_bufend
	cpi	yl, low(CNTBUFEND)
	brlo	calc_count_nobufend

calc_count_bufend:
	ldi	yh, high(CNTBUF)	; load begin of counter buffer
	ldi	yl, low(CNTBUF)

calc_count_nobufend:
	sub	fr0, d			; calculate counter value difference
	sbc	fr1, c
	sbc	fr2, b
	sbc	fr3, a
	ldi	a, 32
	mov	fre, a			; set exponent

	mov	a, flags		; load flags
	andi	a, F_PRESCALE		; sheck if prescaler input activated
	breq	calc_count_noprescale	; if not, were done - otherwise...

	ldi	a, prescale		; get prescale compensation shifts
	add	fre, a			; add to exponent
	call	conv32f8toi		; convert to integer (exponent 32)

calc_count_noprescale:
	call	show			; convert count to decimal & show

	ret

;===== variables =======================================================

.dseg 

CNTBUF:		.byte rate * 4		; counter buffer
CNTBUFEND:
