----------------------------------------------------------------------------------
-- la.vhd
--
-- Copyright (C) 2006 Michael Poppitz
-- 
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or (at
-- your option) any later version.
--
-- This program is distributed in the hope that it will be useful, but
-- WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along
-- with this program; if not, write to the Free Software Foundation, Inc.,
-- 51 Franklin St, Fifth Floor, Boston, MA 02110, USA
--
----------------------------------------------------------------------------------
--
-- Details: http://sump.org/projects/analyzer/
--
-- Logic Analyzer top level module. It connects all the other modules
-- and defines all inputs and outputs that represent phyisical pins of
-- the fpga.
--
-- It defines two constants FREQ and RATE. The first is the clock frequency 
-- used for receiver and transmitter for generating the proper baud rate.
-- The second defines the speed at which to operate the serial port.
--
----------------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity la is
	Port(
		clockIn : in std_logic;
	   resetSwitch : in std_logic;
		input : in std_logic_vector(31 downto 0);
		rx : in std_logic;
		tx : inout std_logic;

		an : OUT std_logic_vector(3 downto 0);
		segment : OUT std_logic_vector(7 downto 0);
		led : OUT std_logic_vector(7 downto 0);

		switch : in std_logic_vector(1 downto 0);

		ramIO1 : INOUT std_logic_vector(15 downto 0);
		ramIO2 : INOUT std_logic_vector(15 downto 0);      
		ramA : OUT std_logic_vector(17 downto 0);
		ramWE : OUT std_logic;
		ramOE : OUT std_logic;
		ramCE1 : OUT std_logic;
		ramUB1 : OUT std_logic;
		ramLB1 : OUT std_logic;
		ramCE2 : OUT std_logic;
		ramUB2 : OUT std_logic;
		ramLB2 : OUT std_logic
	);
end la;

architecture Behavioral of la is

	COMPONENT clockman
	PORT(
		clkin : IN std_logic;       
		clk0 : OUT std_logic;
		clk180 : OUT std_logic
		);
	END COMPONENT;
	
	COMPONENT receiver
	generic (
		FREQ : integer;
		RATE : integer
	);
	PORT(
		rx : IN std_logic;
		clock : IN std_logic;    
	   reset : in STD_LOGIC;
		op : out std_logic_vector(7 downto 0);
		data : out std_logic_vector(31 downto 0);
	   execute : out STD_LOGIC
	   );
	END COMPONENT;

	COMPONENT decoder
	PORT ( opcode : in  STD_LOGIC_VECTOR (7 downto 0);
			 execute : in std_logic;
			  clock : in std_logic;
           wrtrigmask : out  STD_LOGIC;
           wrtrigval : out  STD_LOGIC;
			  wrspeed : out STD_LOGIC;
			  wrsize : out std_logic;
			  wrFlags : out std_logic;
			  arm : out std_logic;
				reset : out std_logic;
				xon : out std_logic;
				xoff : out std_logic
		);
	END COMPONENT;

	COMPONENT flags
	PORT(
		data : IN std_logic_vector(5 downto 0);
		clock : IN std_logic;
		write : IN std_logic;          
		demux : OUT std_logic;
	   filter : OUT std_logic;
		disabledGroups : out std_logic_vector(3 downto 0)
	);
	END COMPONENT;
	
	COMPONENT demux
	PORT(
		input : IN std_logic_vector(15 downto 0);
		clock : IN std_logic;
		clock180 : IN std_logic;
		output : OUT std_logic_vector(31 downto 0)
		);
	END COMPONENT;

	COMPONENT filter
	PORT(
		input : IN std_logic_vector(31 downto 0);
		clock : IN std_logic;
		clock180 : IN std_logic;
		output : OUT std_logic_vector(31 downto 0)
		);
	END COMPONENT;

	COMPONENT sampler
	PORT(
		input : IN std_logic_vector(31 downto 0);
		clock : IN std_logic;
		data : IN std_logic_vector(23 downto 0);
		wrDivider : IN std_logic;          
		sample : OUT std_logic_vector(31 downto 0);
		ready : OUT std_logic
		);
	END COMPONENT;
	
	COMPONENT trigger
	PORT(
		input : IN std_logic_vector(31 downto 0);
		data : IN std_logic_vector(31 downto 0);
	   clock : in std_logic;
		reset : in std_logic;
		wrMask : IN std_logic;
		wrValue : IN std_logic;
		arm : IN std_logic;
		demuxed : in std_logic;
		run : OUT std_logic
		);
	END COMPONENT;

	COMPONENT controller
	PORT(
		clock : IN std_logic;
		reset : in std_logic;
		input : IN std_logic_vector(31 downto 0);    
	   inputReady : in std_logic;
		data : in std_logic_vector(31 downto 0);
		wrSize : in std_logic;
		run : in std_logic;
		txBusy : in std_logic;
		send : inout std_logic;
		output : out std_logic_vector(31 downto 0);    
		ramIO1 : INOUT std_logic_vector(15 downto 0);
		ramIO2 : INOUT std_logic_vector(15 downto 0);      
		ramA : OUT std_logic_vector(17 downto 0);
		ramWE : OUT std_logic;
		ramOE : OUT std_logic;
		ramCE1 : OUT std_logic;
		ramUB1 : OUT std_logic;
		ramLB1 : OUT std_logic;
		ramCE2 : OUT std_logic;
		ramUB2 : OUT std_logic;
		ramLB2 : OUT std_logic
		);
	END COMPONENT;

	COMPONENT transmitter
	generic (
		FREQ : integer;
		RATE : integer
	);
	PORT(
		data : IN std_logic_vector(31 downto 0);
		disabledGroups : in std_logic_vector (3 downto 0);
		write : IN std_logic;
		xon : in std_logic;
		xoff : in std_logic;
		clock : IN std_logic;
		reset : in std_logic;
		tx : OUT std_logic;
		busy : out std_logic;
		pause : out std_logic
		);
	END COMPONENT;

	COMPONENT display
	PORT(
		data : IN std_logic_vector(31 downto 0);
		clock : IN std_logic;          
		an : OUT std_logic_vector(3 downto 0);
		segment : OUT std_logic_vector(7 downto 0)
		);
	END COMPONENT;

signal opcode : std_logic_vector (7 downto 0);
signal data : std_logic_vector (31 downto 0);
signal execute : std_logic;
signal displayData : std_logic_vector (31 downto 0);
signal output : std_logic_vector (31 downto 0);
signal filteredInput, demuxedInput, synchronizedInput, selectedInput, sample : std_logic_vector (31 downto 0);
signal wrtrigmask, wrtrigval, wrDivider, wrsize, run, arm, txBusy, send, xon, xoff, pause : std_logic;
signal clock, clock180, reset, resetCmd, flagDemux, flagFilter, wrFlags, sampleReady: std_logic;
signal flagDisabledGroups : std_logic_vector(3 downto 0);

constant FREQ : integer := 100000000;
constant RATE : integer := 115200;

begin
	-- switches and leds are kept in design to be available for debugging
	led(7 downto 0) <= "000" & switch & (pause, tx, rx);
	displayData <= output;

	-- reset is triggered either by switch or reset command
	reset <= resetSwitch or resetCmd;

	-- synch input guarantees use of iob ff on spartan 3 (as filter and demux do)
	process (clock)
	begin
		if rising_edge(clock) then
			synchronizedInput <= input;
		end if;
	end process;

	-- add another pipeline step for input selector to not decrease maximum clock
	process (clock) 
	begin
		if rising_edge(clock) then
			if flagDemux = '1' then
				selectedInput <= demuxedInput;
			else
				if flagFilter = '1' then
					selectedInput <= filteredInput;
				else
					selectedInput <= synchronizedInput;
				end if;
			end if;
		end if;
	end process;
	
	Inst_clockman: clockman PORT MAP(
		clkin => clockIn,
		clk0 => clock,
		clk180 => clock180
	);
	
	Inst_receiver: receiver
	generic map (
		FREQ => FREQ,
		RATE => RATE
	)
	PORT MAP(
		rx => rx,
		clock => clock,
		reset => reset,
		op => opcode,
		data => data,
		execute => execute
	);

	Inst_decoder: decoder PORT MAP(
		opcode => opcode,
		execute => execute,
		clock => clock,
		wrtrigmask => wrtrigmask,
		wrtrigval => wrtrigval,
		wrspeed => wrDivider,
		wrsize => wrsize,
		wrFlags => wrFlags,
		arm => arm,
		reset => resetCmd,
		xon => xon,
		xoff => xoff
	);

	Inst_flags: flags PORT MAP(
		data => data(5 downto 0),
		clock => clock,
		write => wrFlags,
		demux => flagDemux,
		filter => flagFilter,
		disabledGroups => flagDisabledGroups
	);
	
	Inst_demux: demux PORT MAP(
		input => input(15 downto 0),
		clock => clock,
		clock180 => clock180,
		output => demuxedInput
	);

	Inst_filter: filter PORT MAP(
		input => input,
		clock => clock,
		clock180 => clock180,
		output => filteredInput
	);
	
	Inst_sampler: sampler PORT MAP(
		input => selectedInput,
		clock => clock,
		data => data(23 downto 0),
		wrDivider => wrDivider,
		sample => sample,
		ready => sampleReady
	);
	
	Inst_trigger: trigger PORT MAP(
		input => sample,
		data => data,
		clock => clock,
		reset => reset,
		wrMask => wrtrigmask,
		wrValue => wrtrigval,
		arm => arm,
		demuxed => flagDemux,
		run => run
	);

	Inst_controller: controller PORT MAP(
		clock => clock,
		reset => reset,
		input => sample,
		inputReady => sampleReady,
		data => data,
		wrSize => wrsize,
		run => run,
		txBusy => txBusy,
		send => send,
		output => output,
		ramA => ramA,
		ramWE => ramWE,
		ramOE => ramOE,
		ramIO1 => ramIO1,
		ramCE1 => ramCE1,
		ramUB1 => ramUB1,
		ramLB1 => ramLB1,
		ramIO2 => ramIO2,
		ramCE2 => ramCE2,
		ramUB2 => ramUB2,
		ramLB2 => ramLB2 
	);

	Inst_transmitter: transmitter
	generic map (
		FREQ => FREQ,
		RATE => RATE
	)
	PORT MAP(
		data => output,
		disabledGroups => flagDisabledGroups,
		write => send,
		xon => xon,
		xoff => xoff,
		clock => clock,
		reset => reset,
		tx => tx,
		busy => txBusy,
		pause => pause
	);
	
	Inst_display: display PORT MAP(
		data => displayData,
		clock => clock,
		an => an,
		segment => segment
	);

end Behavioral;

