/*
 *  Copyright (C) 2006 Michael Poppitz
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or (at
 *  your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin St, Fifth Floor, Boston, MA 02110, USA
 *
 */
package org.sump.analyzer;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;

import javax.swing.JComponent;
/**
 * This component displays a logic level diagram which is obtained from a {@link CapturedData} object.
 * <p>
 * Component size changes with the size of the diagram.
 * Therefore it should only be used from within a JScrollPane.
 *
 * @version 0.5
 * @author Michael "Mr. Sump" Poppitz
 *
 */
public class Diagram extends JComponent implements MouseMotionListener {

	/**
	 * Create a new empty diagram to be placed in a container.
	 *
	 */
	public Diagram() {
		super();
		
		this.size = new Dimension(25, 1);
		
		this.signal = new Color(0,0,196);
		this.trigger = new Color(196,255,196);
		this.grid = new Color(196,196,196);
		this.text = new Color(0,0,0);
		this.time = new Color(0,0,0);
		this.groupBackground = new Color(242,242,242);
		this.background = new Color(255,255,255);
		
		this.offsetX = 25;
		this.offsetY = 18;
		
		zoomDefault();
		setBackground(background);

		this.addMouseMotionListener(this);

		this.capturedData = null;
	}
	
	/**
	 * Resizes the diagram as required by available data and scaling factor.
	 *
	 */
	private void resize() {
		if (capturedData == null)
			return;
	
		int[] data = capturedData.values;
		Rectangle rect = getBounds();
		rect.setSize((int)(25 + scale * data.length), rect.getSize().height);
		setBounds(rect);
		size.width = (int)(25 + scale * data.length);
		size.height = 20 + 20 * (capturedData.channels + capturedData.channels / 8);
		update(this.getGraphics());
	}
	
	/**
	 * Sets the captured data object to use for drawing the diagram.
	 * 
	 * @param capturedData		captured data to base diagram on
	 */
	public void setCapturedData(CapturedData capturedData) {
		this.capturedData = capturedData;
		
		float step = (100 / scale) / capturedData.rate;
		
		unitFactor = 1;
		unitName = "s";
		if (step <= 0.000001) { unitFactor = 1000000000; unitName = "ns"; } 
		else if (step <= 0.001) { unitFactor = 1000000; unitName = "µs"; } 
		else if (step <= 1) { unitFactor = 1000; unitName = "ms"; } 

		resize();
	}

	/**
	 * Returns the captured data object currently displayed in the diagram.
	 * 
	 * @return diagram's current captured data
	 */
	public CapturedData getCapturedData() {
		return (capturedData);
	}

	
	/**
	 * Zooms in by factor 2 and resizes the component accordingly.
	 *
	 */
	public void zoomIn() {
		if (scale < 20) {
			scale = scale * 2;
			resize();
		}
	}
	
	/**
	 * Zooms out by factor 2 and resizes the component accordingly.
	 *
	 */
	public void zoomOut() {
		scale = scale / 2;
		resize();
	}
	
	/**
	 * Reverts back to the standard zoom level.
	 *
	 */
	public void zoomDefault() {
		scale = 10;
		resize();
	}

	/**
	 * Gets the dimensions of the full diagram.
	 * Used to inform the container (preferrably a JScrollPane) about the size.
	 */
	public Dimension getPreferredSize() {
		return (size);
	}
	
	/**
	 * Gets the dimensions of the full diagram.
	 * Used to inform the container (preferrably a JScrollPane) about the size.
	 */
	public Dimension getMinimumSize() {
		return (size);
	}

	private void drawEdge(Graphics g, int x, int y, boolean falling, boolean rising) {
		if (scale <= 1) {
			g.drawLine(x, y, x, y + 14);
		} else {
			int edgeX = x;
			if (scale >= 5)
				edgeX += (int)(scale * 0.4);
	
			if (rising) {
				g.drawLine(x, y + 14, edgeX, y);
				g.drawLine(edgeX, y, x + (int)scale, y);
			}	
			if (falling) {
				g.drawLine(x, y, edgeX, y + 14);
				g.drawLine(edgeX, y + 14, x + (int)scale, y + 14);
			}	
		}
	}
	/**
	 * Draws a channel.
	 * @param g graphics context to draw on
	 * @param x x offset
	 * @param y y offset
	 * @param data array containing the sampled data
	 * @param n number of channel to display
	 * @param from index of first sample to display
	 * @param to index of last sample to display
	 */
	private void drawChannel(Graphics g, int x, int y, int[] data, int n, int from, int to) {
		for (int current = from; current < to;) {
			int currentX = (int)(x + current * scale);
			int currentV = (data[current] >> n) & 0x01;
			int nextV = currentV;
			int next = current;
	
			// scan for the next change
			do {
				nextV = (data[++next] >> n) & 0x01;
			} while ((next < to) && (nextV == currentV));
			int currentEndX = currentX + (int)(scale * (next - current - 1));
			
			// draw straight line up to the point of change and a edge if not at end
			if (currentV == nextV) {
				g.drawLine(currentX, y + 14 * (1 - currentV), currentEndX + (int)scale, y + 14 * (1 - currentV));
			} else {
				g.drawLine(currentX, y + 14 * (1 - currentV), currentEndX, y + 14 * (1 - currentV));
				if (currentV > nextV)
					drawEdge(g, currentEndX, y, true, false);
				else if (currentV < nextV)
					drawEdge(g, currentEndX, y, false, true);
			}
			current = next;
		}
	}
	
	/**
	 * Draws a group of 8 channels.
	 * @param g graphics context to draw on
	 * @param x x offset
	 * @param y y offset
	 * @param data array containing the sampled data
	 * @param n number of group to display (0-3 for 32 channels)
	 * @param from index of first sample to display
	 * @param to index of last sample to display
	 */
	private void drawGroup(Graphics g, int x, int y, int[] data, int n, int from, int to) {
		for (int current = from; current < to;) {
			int currentX = (int)(x + current * scale);
			int currentXSpace = (int)(x + (current - 1) * scale);
			int currentV = (data[current] >> (8 * n)) & 0xff;
			int nextV = currentV;
			int next = current;

			// scan for the next change
			do {
				nextV = (data[++next] >> (8 * n)) & 0xff;
			} while ((next < to) && (nextV == currentV));
			int currentEndX = currentX + (int)(scale * (next - current - 1));
			
			// draw straight lines up to the point of change and a edge if not at end
			if (currentV == nextV) {
				g.drawLine(currentX, y + 14, currentEndX + (int)scale, y + 14);
				g.drawLine(currentX, y, currentEndX + (int)scale, y);
			} else {
				g.drawLine(currentX, y + 14, currentEndX, y + 14);
				g.drawLine(currentX, y, currentEndX, y);
				drawEdge(g, currentEndX, y, true, true);
			}
			
			// if steady long enough, add hex value
			if (currentEndX - currentXSpace > 15) {
				if (currentV >= 0x10)
					g.drawString(Integer.toString(currentV, 16), (currentXSpace + currentEndX) / 2 - 2, y + 12);
				else
					g.drawString("0" + Integer.toString(currentV, 16), (currentXSpace + currentEndX) / 2 - 2, y + 12);

			}
			
			current = next;
		}
	}
	
	/**
	 * Convert x position to sample index.
	 * @param x horizontal position in pixels
	 * @return sample index
	 */
	private int xToIndex(int x) {
		int index = (int)((x - offsetX) / scale);
		if (index < 0)
			index = 0;
		if (index >= capturedData.values.length)
			index = capturedData.values.length - 1;
		return (index);
	}
	
	/**
	 * Convert sample count to time string.
	 * @param count sample count (or index)
	 * @return string containing time information
	 */
	private String indexToTime(int count) {
		return (((count * unitFactor) / capturedData.rate) + unitName);
	}

	/**
	 * Paints the diagram to the extend necessary.
	 */
	public void paintComponent(Graphics g) {
		if (capturedData == null)
			return;
		
		int[] data = capturedData.values;
		int triggerPosition = capturedData.triggerPosition;
		int rate = capturedData.rate;
		int channels = capturedData.channels;
		int enabled = capturedData.enabledChannels;
		
		int xofs = offsetX;
		int yofs = offsetY + 2;

		// obtain portion of graphics that needs to be drawn
		Rectangle clipArea = g.getClipBounds();

		// find index of first row that needs drawing
		int firstRow = xToIndex(clipArea.x);
		if (firstRow < 0)
			firstRow = 0;
			
		// find index of last row that needs drawing
		int lastRow = xToIndex(clipArea.x + clipArea.width) + 1;
		if (lastRow >= data.length)
 			lastRow = data.length - 1;

		// paint portion of background that needs drawing
		g.setColor(background);
		g.fillRect(clipArea.x, clipArea.y, clipArea.width, clipArea.height);
		g.setColor(groupBackground);
		for (int block = 0; block < channels / 8; block++) {
			int bofs = 20 * 9 * block + yofs;
			g.fillRect(clipArea.x, 20 * 8 + bofs - 2, clipArea.width, 19);
		}

		// draw trigger if existing and visible
		if (triggerPosition >= firstRow && triggerPosition <= lastRow) {
			g.setColor(trigger);
			g.fillRect(xofs + (int)(triggerPosition * scale) - 1, 0, (int)(scale) + 2, yofs + 36 * 20);		
		}
		
		// draw time line
		if (rate > 0) {
			int rowInc = (int)(100 / scale);

			g.setColor(time);
			for (int row = (firstRow / rowInc) * rowInc; row < lastRow; row += rowInc) {
				int pos = (int)(xofs + scale * row);
				g.drawLine(pos, 1, pos, 15);
				g.drawString((Math.round(10 * ((row - triggerPosition) * unitFactor) / (float)rate) / 10F) + unitName, pos + 5, 10);
				int divs = 10;
				if (scale > 10) divs = 5;
				for (int sub = rowInc / divs; sub < rowInc; sub += rowInc / divs)
					g.drawLine(pos + (int)(sub * scale), 12, pos + (int)(sub * scale), 15);
			}
		}

		for (int block = 0; block < channels / 8; block++) {
			int bofs = 20 * 9 * block + yofs;

			// draw channel separators
			for (int bit = 0; bit < 8; bit++) {
				g.setColor(grid);
				g.drawLine(clipArea.x, 20 * bit + bofs + 17, clipArea.x + clipArea.width, 20 * bit + bofs + 17);
				g.setColor(text);
				g.drawString("" + (bit + block * 8), 5, 20 * bit + bofs + 12);
			}
			g.setColor(grid);
			g.drawLine(clipArea.x, 20 * 8 + bofs + 17, clipArea.x + clipArea.width, 20 * 8 + bofs + 17);
			g.setColor(text);
			g.drawString("B" + block, 5, 20 * 8 + bofs + 12);
			
			// draw actual data
			g.setColor(signal);
			for (int bit = block * 8; bit < block * 8 + 8; bit++)
				if (((enabled >> bit) & 1) == 1)
					drawChannel(g, xofs, yofs + 20 * (bit + block), data, bit, firstRow, lastRow);
			if (((enabled >> (block * 8)) & 0xff) == 0xff)
				drawGroup(g, xofs, yofs + 20 * (8 + 9 * block), data, block, firstRow, lastRow);
		}
	}

	/**
	 * Update status information.
	 * Notifies {@link StatusChangeListener}.
	 * @param dragging <code>true</code> indicates that dragging information should be added 
	 */
	private void updateStatus(boolean dragging) {
		if (capturedData == null || statusChangeListener == null)
			return;

		StringBuffer sb = new StringBuffer(" ");
		
		int row = (mouseY - offsetY) / 20;
		if (row <= capturedData.channels + (capturedData.channels / 9)) {
			if (row % 9 == 8)
				sb.append("Byte " + (row / 9));
			else
				sb.append("Channel " + (row - (row / 9)));
			sb.append(" | ");
		}

		if (dragging && xToIndex(mouseDragX) != xToIndex(mouseX)) {
			int index = xToIndex(mouseDragX);
			float frequency = Math.abs(capturedData.rate / (index - xToIndex(mouseX)));
			String unit;
			int div;
			if (frequency >= 1000000) { unit = "MHz"; div = 1000000; }
			else if (frequency >= 1000) { unit = "kHz"; div = 1000; }
			else { unit = "Hz"; div = 1; } 
			
			sb.append("Time " + indexToTime(index - capturedData.triggerPosition));
			sb.append(" (Duration " + indexToTime(index - xToIndex(mouseX)) + ", ");
			sb.append("Frequency " + (frequency / (float)div) + unit + ")");
		} else {
			sb.append("Time " + indexToTime(xToIndex(mouseX) - capturedData.triggerPosition));
		}
		statusChangeListener.statusChanged(sb.toString());
	}
	
	/**
	 * Handles mouse dragged events and produces status change "events" accordingly.
	 */
	public void mouseDragged(MouseEvent event) {
		mouseDragX = event.getX();
		updateStatus(true);
	}

	/**
	 * Handles mouse moved events and produces status change "events" accordingly.
	 */
	public void mouseMoved(MouseEvent event) {
		mouseX = event.getX();
		mouseY = event.getY();
		updateStatus(false);
	}

	/**
	 * Adds a status change listener for this diagram,
	 * Simple implementation that will only call the last added listener on status change.
	 */
	public void addStatusChangeListener(StatusChangeListener listener) {
		statusChangeListener = listener;
	}

	private CapturedData capturedData;
	private long unitFactor;
	private String unitName;
	
	private int offsetX;
	private int offsetY;
	private int mouseX;
	private int mouseY;
	private int mouseDragX;
	private StatusChangeListener statusChangeListener;
	
	private float scale;
	
	private Color signal;
	private Color trigger;
	private Color grid;
	private Color text;
	private Color time;
	private Color groupBackground;
	private Color background;
	
	private Dimension size;

	private static final long serialVersionUID = 1L;
}
