/*
 *  Copyright (C) 2006 Michael Poppitz
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or (at
 *  your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin St, Fifth Floor, Boston, MA 02110, USA
 *
 */
package org.sump.analyzer;

import gnu.io.CommPortIdentifier;
import gnu.io.SerialPort;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Enumeration;
import java.util.LinkedList;

/**
 * Device provides access to the physical logic analyzer device.
 * It requires the rxtx package from http://www.rxtx.org/ to
 * access the serial port the analyzer is connected to.
 * 
 * @version 0.5
 * @author Michael "Mr. Sump" Poppitz
 *
 */
public class Device extends Object {
	/** set trigger mask */
	private final static int SETTRIGMASK = 0xc0;
	/** set trigger value */
	private final static int SETTRIGVAL = 0xc1;
	/** set clock divider */
	private final static int SETDIVIDER = 0x80;
	/** set sample counters */
	private final static int SETSIZE = 0x81;
	/** set flags */
	private final static int SETFLAGS = 0x82;

	/** reset analyzer */
	private final static int RESET = 0x00;
	/** arm trigger / run device */
	private final static int RUN = 0x01;
	/** continue data transmission to host */
	private final static int XON = 0x11;
	/** pause data transmission to host */
	private final static int XOFF = 0x13;

	
	private final static int FLAG_DEMUX = 0x00000001;		// demultiplex
	private final static int FLAG_FILTER = 0x00000002;		// noise filter
	private final static int FLAG_DISABLE_G0 = 0x00000004;	// disable channel group 0
	private final static int FLAG_DISABLE_G1 = 0x00000008;	// disable channel group 1
	private final static int FLAG_DISABLE_G2 = 0x00000010;	// disable channel group 2
	private final static int FLAG_DISABLE_G3 = 0x00000020;	// disable channel group 3

	private final static int CLOCK = 100000000;	// device clock in Hz
	
	/**
	 * Creates a device object.
	 *
	 */
	public Device() {
		triggerMask = 0;
		triggerValue = 0;
		triggerEnabled = false;
		filterEnabled = false;
		demux = true;
		divider = 0;
		stopCounter = 6400;
		readCounter = 12800;
		enabledGroups = new boolean[4];
		setEnabledChannels(-1); // enable all channels
		
		percentageDone = -1;
		
		port = null;
	}

	/**
	 * Sets the number of samples to obtain when started.
	 * 
	 * @param size number of samples, must be between 4 and 256*1024
	 */
	public void setSize(int size) {
		double ratio = (double)stopCounter / (double)readCounter;
		readCounter = size;
		setRatio(ratio);
	}
	
	/**
	 * Sets the ratio for samples to read before and after started.
	 * @param ratio	value between 0 and 1; 0 means all before start, 1 all after
	 */
	public void setRatio(double ratio) {
		stopCounter = (int)(readCounter * ratio);
	}

	/**
	 * Set the sampling rate.
	 * All rates must be a divisor of 200.000.000.
	 * Other rates will be adjusted to a matching divisor.
	 * 
	 * @param rate		sampling rate in Hz
	 */
	public void setRate(int rate) {
		if (rate > CLOCK) {
			demux = true;
			divider = (2 * CLOCK / rate) - 1;
		} else {
			demux = false;
			divider = (CLOCK / rate) - 1;
		}
	}
	
	/**
	 * Configures the conditions that must be met to fire the trigger.
	 * <br>
	 * Each bit of the integer parameters represents one channel.
	 * <br>
	 * The LSB represents channel 0, the MSB channel 31.
	 * <p>
	 * To disable the trigger, set mask to 0. This will cause it to always fire.
	 * 
	 * @param mask bit map defining which channels to watch
	 * @param value bit map defining what value to wait for on watched channels
	 */
	public void setTrigger(int mask, int value) {
		if (!demux) {
			triggerMask = mask;
			triggerValue = value;
		} else {
			triggerMask = mask & 0xffff;
			triggerValue = value & 0xffff;
			triggerMask |= triggerMask << 16;
			triggerValue |= triggerValue << 16;
		}
	}
	
	/**
	 * Sets wheter or not to enable the trigger.
	 * @param enable <code>true</code> enables the trigger, <code>false</code> disables it.
	 */
	public void setTriggerEnabled(boolean enable) {
		triggerEnabled = enable;
	}

	/**
	 * Sets wheter or not to enable the noise filter.
	 * @param enable <code>true</code> enables the noise filter, <code>false</code> disables it.
	 */
	public void setFilterEnabled(boolean enable) {
		filterEnabled = enable;
	}
	
	/**
	 * Set enabled channels.
	 * @param bit map defining enabled channels
	 */
	public void setEnabledChannels(int mask) {
		enabledChannels = mask;
		// determine enabled groups
		for (int i = 0; i < 4; i++)
			enabledGroups[i] = ((enabledChannels  >> (8 * i)) & 0xff) > 0;
	}


	/**
	 * Get the maximum sampling rate available.
	 * @return maximum sampling rate
	 */
	public int getMaximumRate() {
		return (2 * CLOCK);
	}

	/**
	 * Returns the current trigger mask.
	 * @return current trigger mask
	 */
	public int getTriggerMask() {
		return (triggerMask);
	}

	/**
	 * Returns the current trigger value.
	 * @return current trigger value
	 */
	public int getTriggerValue() {
		return (triggerValue);
	}

	/**
	 * Returns the currently enabled channels.
	 * @return bitmask with enabled channels represented as 1
	 */
	public int getEnabledChannels() {
		return (enabledChannels);
	}

	/**
	 * Returns wether or not the trigger is enabled.
	 * @return <code>true</code> when trigger is enabled, <code>false</code> otherwise
	 */
	public boolean isTriggerEnabled() {
		return (triggerEnabled);
	}

	/**
	 * Returns wether or not the noise filter is enabled.
	 * @return <code>true</code> when noise filter is enabled, <code>false</code> otherwise
	 */
	public boolean isFilterEnabled() {
		return (filterEnabled);
	}

	/**
	 * Returns wether or not the noise filter can be used in the current configuration.
	 * @return <code>true</code> when noise filter is available, <code>false</code> otherwise
	 */
	public boolean isFilterAvailable() {
		return (!demux);
	}

	/**
	 * Returns the number of available channels in current configuration.
	 * @return number of available channels
	 */
	public int getAvailableChannelCount() {
		if (demux)
			return (16);
		else
			return (32);
	}

	/**
	 * Returns wether or not the device is currently running.
	 * It is running, when another thread is inside the run() method reading data from the serial port.
	 * @return <code>true</code> when running, <code>false</code> otherwise
	 */
	public boolean isRunning() {
		return (percentageDone != -1);
	}

	/**
	 * Returns the percentage of the expected data that has already been read.
	 * The return value is only valid when <code>isRunning()</code> returns <code>true</code>. 
	 * @return percentage already read (0-100)
	 */
	public int getPercentage() {
		return (percentageDone);
	}

	/**
	 * Gets a string array containing the names all available serial ports.
	 * @return array containing serial port names
	 */
	static public String[] getPorts() {
		Enumeration portIdentifiers = CommPortIdentifier.getPortIdentifiers();
		LinkedList portList = new LinkedList();
		CommPortIdentifier portId = null;

		while (portIdentifiers.hasMoreElements()) {
			portId = (CommPortIdentifier) portIdentifiers.nextElement();
			if (portId.getPortType() == CommPortIdentifier.PORT_SERIAL) {
				portList.addLast(portId.getName());
				System.out.println(portId.getName());
			}
		}
			
		return ((String[])portList.toArray(new String[1]));
	}

	/**
	 * Attaches the given serial port to the device object.
	 * The method will try to open the port.
	 * <p>
	 * A return value of <code>true</code> does not guarantee that a
	 * logic analyzer is actually attached to the port.
	 * <p>
	 * If the device is already attached to a port this port will be
	 * detached automatically. It is therefore not necessary to manually
	 * call <code>detach()</code> before reattaching.
	 *
	 * @param portName		the name of the port to open
	 * @return				<code>true</code> when the port has been assigned successfully;
	 * 						<code>false</code> otherwise.
	 */
	public boolean attach(String portName) {
		Enumeration portList = CommPortIdentifier.getPortIdentifiers();
		CommPortIdentifier portId = null;
		boolean found = false;

		try {
			detach();
	
			while (!found && portList.hasMoreElements()) {
				portId = (CommPortIdentifier) portList.nextElement();
	
				if (portId.getPortType() == CommPortIdentifier.PORT_SERIAL) {
					if (portId.getName().equals(portName)) {
						found = true;
					}
				}
			}
			
			if (found) {
				port = (SerialPort) portId.open("Logic Analyzer Client", 1000);
				
				port.setSerialPortParams(
					115200,
					SerialPort.DATABITS_8,
					SerialPort.STOPBITS_1,
					SerialPort.PARITY_NONE
				);
				port.setFlowControlMode(SerialPort.FLOWCONTROL_XONXOFF_IN);
				port.disableReceiveFraming();
				port.enableReceiveTimeout(1000);
			}
		} catch(Exception E) {
			E.printStackTrace(System.out);
			return (false);
		}		
		return (found);
	}
	
	/**
	 * Detaches the currently attached port, if one exists.
	 * This will close the serial port.
	 *
	 */
	public void detach() {
		if (port != null) {
			try {
				// try to make sure device is reset (see run() for loop explanation)
				for (int i = 0; i < 5; i++)
					sendCommand(port.getOutputStream(), RESET);
			} catch (IOException E) {}
			port.close();
		}
	}
	
	/**
	 * Sends a long command to the given stream.
	 * 
	 * @param output	stream to write command to
	 * @param opcode	one byte operation code
	 * @param data		four byte data portion
	 * @throws IOException if writing to stream fails
	 */
	private void sendCommand(OutputStream output, int opcode, int data) throws IOException {
		byte[] raw = new byte[5];
		int mask = 0xff;
		int shift = 0;
		
		raw[0] = (byte)opcode;
		for (int i = 1; i < 5; i++) {
			raw[i] = (byte)((data & mask) >> shift);
			mask = mask << 8;
			shift += 8;
		}

		String debugCmd = "";
		for (int j = 0; j < 5; j++) {
			for (int i = 7; i >= 0; i--) {
				if ((raw[j] & (1 << i)) != 0)
					debugCmd += "1";
				else
					debugCmd += "0";
			}
			debugCmd += " ";
		}
		System.out.println(debugCmd);
		
		output.write(raw);
	}
	
	/**
	 * Sends a short command to the given stream.
	 * 
	 * This method is intended to be used for short commands, but can also be called
	 * with long command opcodes if the data portion is to be set to 0.
	 * 
	 * @param output	stream to write command to
	 * @param opcode	one byte operation code
	 * @throws IOException if writing to stream fails
	 */
	private void sendCommand(OutputStream output, int opcode) throws IOException {
		byte raw = (byte)opcode;
		
		String debugCmd = "";
		for (int i = 7; i >= 0; i--) {
			if ((raw & (1 << i)) != 0)
				debugCmd += "1";
			else
				debugCmd += "0";
		}
		System.out.println(debugCmd);

		output.write(raw);
	}

	/**
	 * Reads <code>channels</code> / 8 bytes from stream and compiles them into a single integer.
	 * 
	 * @param input stream to read from
	 * @param channels number of channels to read (must be multiple of 8)
	 * @return	integer containing four bytes read
	 * @throws IOException if stream reading fails
	 */
	private int readSample(InputStream input, int channels) throws IOException, InterruptedException {
		int v, value = 0;

		for (int i = 0; i < channels / 8; i++) {
			if (enabledGroups[i]) {
				v = input.read();
			} else {
				v = 0;
			}
			if (v < 0 || Thread.interrupted())
				throw new InterruptedException("Data readout interrupted.");
			value |= v << (8 * i);
		}

		return (value);
	}

	/**
	 * Sends the configuration to the device, starts it, reads the captured data
	 * and returns a CapturedData object containing the data read as well as device configuration information.
	 * @return captured data
	 * @throws IOException when writing to or reading from device fails
	 */
	public CapturedData run() throws IOException, InterruptedException {
		OutputStream outputStream = port.getOutputStream();
		InputStream inputStream = port.getInputStream();
		// send reset 5 times because in worst case first 4 are interpreted as data of long command
		for (int i = 0; i < 5; i++)
			sendCommand(outputStream, RESET);

		if (triggerEnabled) {
			sendCommand(outputStream, SETTRIGMASK, triggerMask);
			sendCommand(outputStream, SETTRIGVAL, triggerValue);
		} else {
			sendCommand(outputStream, SETTRIGMASK, 0);
			sendCommand(outputStream, SETTRIGVAL, 0);
		}
		sendCommand(outputStream, SETDIVIDER, divider);

		int flags = 0;
		if (demux) {
			flags |= FLAG_DEMUX;
			for (int i = 0; i < 2; i++)
				if (!enabledGroups[i]) {
					flags |= FLAG_DISABLE_G0 << i;
					flags |= FLAG_DISABLE_G2 << i;
				}
			sendCommand(outputStream, SETSIZE, (((stopCounter - 8) & 0x7fff8) << 13) | (((readCounter & 0x7fff8) >> 3) - 1));
		} else {
			if (filterEnabled)
				flags |= FLAG_FILTER;
			for (int i = 0; i < 4; i++)
				if (!enabledGroups[i])
					flags |= FLAG_DISABLE_G0 << i;
			sendCommand(outputStream, SETSIZE, (((stopCounter - 4) & 0x3fffc) << 14) | (((readCounter & 0x3fffc) >> 2) - 1));
		}
		sendCommand(outputStream, SETFLAGS, flags);
		sendCommand(outputStream, RUN);

		// check if data needs to be multiplexed
		int channels;
		int samples;
		if (demux) {
			channels = 16;
			samples = (readCounter & 0x7fff8);
		} else {
			channels = 32;
			samples = (readCounter & 0x3fffc);
		}

		int[] buffer = new int[samples];

		// wait for first byte forever (trigger could cause long delay)
		for (boolean wait = true; wait == true;) {
			try {
				buffer[samples - 1] =  readSample(inputStream, channels);
				wait = false;
			} catch (InterruptedException e) {}
		}
		
		// read all other samples
		try {
			for (int i = samples - 2; i >= 0 && true; i--) {
				buffer[i] = readSample(inputStream, channels);
				percentageDone = 100 - (100 * i) / buffer.length;
			}
		} finally {
			percentageDone = -1;
			inputStream.close();
			outputStream.close();
		}
		
		int pos = -1;
		if (triggerEnabled)
			pos = readCounter - stopCounter - 2 - (divider == 0 ? 2 : 0);
		int rate = demux ? 2*CLOCK / (divider + 1) : CLOCK / (divider + 1);
		return (new CapturedData(buffer, pos, rate, channels, enabledChannels));
	}
	
	private SerialPort port;
	
	private int percentageDone;
	
	private boolean demux;
	private boolean filterEnabled;
	private boolean triggerEnabled;
	private int triggerMask;
	private int triggerValue;
	private int enabledChannels;
	private boolean enabledGroups[];
	
	private int divider;
	private int stopCounter;
	private int readCounter;
}
